<?php
/**
 * public/trips/update.php
 * Actualiza un viaje. Soporta reemplazo opcional de hasta 5 archivos (imagen/PDF).
 * Corrige el mapeo: en el form el primer input es "foto1" (no "foto").
 */
session_start();
require_once '../../config/db.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: index.php');
    exit;
}

$id = $_POST['id'] ?? '';
if (!$id) {
    $_SESSION['error'] = "ID de viaje inválido.";
    header('Location: index.php');
    exit;
}

/* ----------------------------------------------------------------------------
 * 1) Traer registro actual (para conservar archivos si no se reemplazan)
 * ------------------------------------------------------------------------- */
$stmtOld = $pdo->prepare("SELECT * FROM trips WHERE id = :id LIMIT 1");
$stmtOld->execute(['id' => $id]);
$old = $stmtOld->fetch(PDO::FETCH_ASSOC);
if (!$old) {
    $_SESSION['error'] = "Viaje no encontrado.";
    header('Location: index.php');
    exit;
}

/* ----------------------------------------------------------------------------
 * 2) Capturar datos del formulario
 * ------------------------------------------------------------------------- */
$origin_id          = $_POST['origin_id']          ?? null;
$destination_id     = $_POST['destination_id']     ?? null;
$driver_id          = $_POST['driver_id']          ?? null;
$fecha_origen       = $_POST['fecha_origen']       ?? null;
$fecha_destino      = $_POST['fecha_destino']      ?? null;
$tipo_transporte    = $_POST['tipo_transporte']    ?? 'Transporte';
$km_recorridos      = $_POST['km_recorridos']      ?? 0;
$toneladas          = $_POST['toneladas']          ?? 0;
$tarifa_real        = $_POST['tarifa_real']        ?? 0;
$tarifa_rendida     = $_POST['tarifa_rendida']     ?? 0;
$estado_viaje       = $_POST['estado_viaje']       ?? 'Activo';
$estado_facturacion = $_POST['estado_facturacion'] ?? 'No Facturado';
$comentario         = $_POST['comentario']         ?? '';
$ctg                = $_POST['ctg']                ?? '';
$ctg                = $ctg !== '' ? preg_replace('/[^A-Za-z0-9-]/', '', trim($ctg)) : null;

/* ----------------------------------------------------------------------------
 * 3) Subida opcional de archivos (imagen/pdf)
 *    - En el FORM los names son: foto1, foto2, foto3, foto4, foto5
 *    - En la BD las columnas son: foto, foto2, foto3, foto4, foto5
 * ------------------------------------------------------------------------- */
function subirOpcional($campoForm) {
    if (empty($_FILES[$campoForm]['name'])) return '';

    $permitidas = ['jpg','jpeg','png','gif','webp','pdf'];
    $nombreOrig = $_FILES[$campoForm]['name'];
    $ext = strtolower(pathinfo($nombreOrig, PATHINFO_EXTENSION));
    if (!in_array($ext, $permitidas, true)) return '';

    $base = pathinfo($nombreOrig, PATHINFO_FILENAME);
    $base = preg_replace('/[^A-Za-z0-9_\-]/', '_', $base);
    $nuevo = time() . '_' . uniqid() . '_' . $base . '.' . $ext;

    $dir = __DIR__ . '/../uploads/';
    if (!is_dir($dir)) @mkdir($dir, 0775, true);

    if (move_uploaded_file($_FILES[$campoForm]['tmp_name'], $dir . $nuevo)) {
        return $nuevo;
    }
    return '';
}

// Mapeo correcto: (form name) -> (columna BD)
$slots = [
    ['form' => 'foto1', 'col' => 'foto'],
    ['form' => 'foto2', 'col' => 'foto2'],
    ['form' => 'foto3', 'col' => 'foto3'],
    ['form' => 'foto4', 'col' => 'foto4'],
    ['form' => 'foto5', 'col' => 'foto5'],
];

$nuevos = [];
foreach ($slots as $s) {
    $subido = subirOpcional($s['form']);
    if ($subido !== '') {
        $nuevos[$s['col']] = $subido;              // hay archivo nuevo
    } else {
        $nuevos[$s['col']] = $old[$s['col']] ?? ''; // conservar anterior
    }
}

/* ----------------------------------------------------------------------------
 * 4) Actualizar viaje (backticks en `CTG` por estar en mayúsculas)
 * ------------------------------------------------------------------------- */
$sql = "UPDATE trips SET
            origin_id = :origin_id,
            destination_id = :destination_id,
            driver_id = :driver_id,
            fecha_origen = :fecha_origen,
            fecha_destino = :fecha_destino,
            tipo_transporte = :tipo_transporte,
            km_recorridos = :km_recorridos,
            `CTG` = :ctg,
            toneladas = :toneladas,
            tarifa_real = :tarifa_real,
            tarifa_rendida = :tarifa_rendida,
            estado_viaje = :estado_viaje,
            estado_facturacion = :estado_facturacion,
            comentario = :comentario,
            foto = :foto,
            foto2 = :foto2,
            foto3 = :foto3,
            foto4 = :foto4,
            foto5 = :foto5,
            updated_at = NOW()
        WHERE id = :id";

$stmt = $pdo->prepare($sql);
$stmt->execute([
    'origin_id'          => $origin_id,
    'destination_id'     => $destination_id,
    'driver_id'          => $driver_id,
    'fecha_origen'       => $fecha_origen,
    'fecha_destino'      => $fecha_destino,
    'tipo_transporte'    => $tipo_transporte,
    'km_recorridos'      => $km_recorridos,
    'ctg'                => $ctg,
    'toneladas'          => $toneladas,
    'tarifa_real'        => $tarifa_real,
    'tarifa_rendida'     => $tarifa_rendida,
    'estado_viaje'       => $estado_viaje,
    'estado_facturacion' => $estado_facturacion,
    'comentario'         => $comentario,
    'foto'               => $nuevos['foto'],
    'foto2'              => $nuevos['foto2'],
    'foto3'              => $nuevos['foto3'],
    'foto4'              => $nuevos['foto4'],
    'foto5'              => $nuevos['foto5'],
    'id'                 => $id
]);

/* ----------------------------------------------------------------------------
 * 5) Log simple (opcional)
 * ------------------------------------------------------------------------- */
try {
    $log = $pdo->prepare("INSERT INTO logs (grupo, tipo, id_modif, user, cambios, fecha)
                          VALUES ('viaje', 2, :id_modif, :user, :cambios, NOW())");
    $log->execute([
        'id_modif' => $id,
        'user'     => $_SESSION['user_id'] ?? 0,
        'cambios'  => json_encode(['actualizado' => array_keys($nuevos)], JSON_UNESCAPED_UNICODE)
    ]);
} catch (Exception $e) {
    // silencioso
}

/* ----------------------------------------------------------------------------
 * 6) Redirigir
 * ------------------------------------------------------------------------- */
$_SESSION['success'] = "¡Viaje actualizado con éxito!";
header('Location: trip.php?id=' . urlencode($id));
exit;