<?php
/**
 * public/trips/store.php
 * Crea o actualiza un viaje con soporte de imágenes + PDF (hasta 5),
 * guarda CTG (letras/números/guiones), asocia cliente y registra estados.
 * + numero_factura (opcional) para búsquedas por factura.
 */
session_start();
require_once '../../config/db.php';

// --- Modo actualización rápida desde fetch (estado de viaje o facturación/pago/numero_factura) ---
if (isset($_POST['action']) && $_POST['action'] === 'update_field') {
    $id    = (int)($_POST['id'] ?? 0);
    $field = $_POST['field'] ?? '';
    $value = $_POST[$field] ?? '';

    // Permitimos actualizar estos campos vía AJAX
    $permitidos = ['estado_viaje', 'estado_facturacion', 'estado_pago', 'numero_factura'];
    if (!in_array($field, $permitidos, true)) {
        http_response_code(400);
        echo "Campo no permitido";
        exit;
    }

    // Normalización si es numero_factura (letras/números/guiones/“/”)
    if ($field === 'numero_factura') {
        $value = strtoupper(trim($value));
        $value = preg_replace('/[^A-Za-z0-9\-\/]/', '', $value);
    }

    try {
        $sql = "UPDATE trips SET $field = :valor, updated_at = NOW() WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['valor' => $value, 'id' => $id]);
        echo "OK";
    } catch (Exception $e) {
        http_response_code(500);
        echo "Error al actualizar";
    }
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: create.php');
    exit;
}

/* ----------------------------------------------------------------------------
 * 1) Captura y normalización de datos del formulario
 * ------------------------------------------------------------------------- */
$trip_id            = $_POST['trip_id']            ?? null; // si viene, es UPDATE
$customer_id        = $_POST['customer_id']        ?? null;
$origin_id          = $_POST['origin_id']          ?? null;
$destination_id     = $_POST['destination_id']     ?? null;
$driver_id          = $_POST['driver_id']          ?? null;
$fecha_origen       = $_POST['fecha_origen']       ?? null;
$fecha_destino      = $_POST['fecha_destino']      ?? null;
$mercaderia         = $_POST['mercaderia']         ?? null;
$tipo_transporte    = $_POST['tipo_transporte']    ?? 'Transporte';
$km_recorridos      = $_POST['km_recorridos']      ?? 0;

/* Toneladas de carga y descarga */
$toneladas_carga    = $_POST['toneladas_carga']    ?? null;
$toneladas          = $_POST['toneladas']          ?? 0;

/* Tarifas y estados */
$tarifa_real        = $_POST['tarifa_real']        ?? 0;
$tarifa_rendida     = $_POST['tarifa_rendida']     ?? 0;
$estado_viaje       = $_POST['estado_viaje']       ?? 'Activo';
$estado_facturacion = $_POST['estado_facturacion'] ?? 'No Facturado';
$estado_pago        = $_POST['estado_pago']        ?? 'No Pagado';

/* Comentarios / resumen */
$comentario         = $_POST['comentario']         ?? '';
$linea_factura      = $_POST['linea_factura']      ?? null;

/* NUEVO: número de factura (opcional) */
$numero_factura     = $_POST['numero_factura']     ?? null;
$numero_factura     = $numero_factura !== null
    ? preg_replace('/[^A-Za-z0-9\-\/]/', '', strtoupper(trim($numero_factura)))
    : null;

/* CTG: permitir letras, números y guiones; sin espacios ni símbolos */
$ctg = $_POST['ctg'] ?? null;
$ctg = $ctg !== null ? preg_replace('/[^A-Za-z0-9-]/', '', trim($ctg)) : null;

/* casteos suaves */
$trip_id         = $trip_id         ? (int)$trip_id : null;
$customer_id     = $customer_id     !== null ? (int)$customer_id     : null;
$origin_id       = $origin_id       !== null ? (int)$origin_id       : null;
$destination_id  = $destination_id  !== null ? (int)$destination_id  : null;
$driver_id       = $driver_id       !== null ? (int)$driver_id       : null;
$km_recorridos   = (int)$km_recorridos;

/* Convertir posibles comas decimales a punto antes de float */
$toneladas_carga = ($toneladas_carga === null || $toneladas_carga === '') ? null : (float)str_replace(',', '.', $toneladas_carga);
$toneladas       = (float)str_replace(',', '.', (string)$toneladas);
$tarifa_real     = (float)str_replace(',', '.', (string)$tarifa_real);
$tarifa_rendida  = (float)str_replace(',', '.', (string)$tarifa_rendida);

/* ----------------------------------------------------------------------------
 * 2) Subida de archivos (imágenes y/o PDF) - inputs: foto1..foto5
 *    Columnas DB: foto, foto2, foto3, foto4, foto5
 * ------------------------------------------------------------------------- */
function guardarArchivoOpcional(string $campo): string
{
    if (empty($_FILES[$campo]['name'])) {
        return '';
    }

    $permitidas = ['jpg','jpeg','png','gif','webp','pdf'];
    $nombreOriginal = $_FILES[$campo]['name'];
    $ext = strtolower(pathinfo($nombreOriginal, PATHINFO_EXTENSION));

    if (!in_array($ext, $permitidas, true)) {
        return '';
    }

    $base = pathinfo($nombreOriginal, PATHINFO_FILENAME);
    $base = preg_replace('/[^A-Za-z0-9_\-]/', '_', $base);

    $nuevoNombre = time() . '_' . uniqid() . '_' . $base . '.' . $ext;

    $carpeta = __DIR__ . '/../uploads/';
    if (!is_dir($carpeta)) {
        @mkdir($carpeta, 0775, true);
    }

    if (move_uploaded_file($_FILES[$campo]['tmp_name'], $carpeta . $nuevoNombre)) {
        return $nuevoNombre;
    }
    return '';
}

$foto  = guardarArchivoOpcional('foto1');
$foto2 = guardarArchivoOpcional('foto2');
$foto3 = guardarArchivoOpcional('foto3');
$foto4 = guardarArchivoOpcional('foto4');
$foto5 = guardarArchivoOpcional('foto5');

/* ----------------------------------------------------------------------------
 * 3) Guardar viaje (INSERT o UPDATE)
 * ------------------------------------------------------------------------- */
try {

    if ($trip_id) {
        // --- UPDATE ---
        $sets = [
            "customer_id = :customer_id",
            "origin_id = :origin_id",
            "destination_id = :destination_id",
            "driver_id = :driver_id",
            "mercaderia = :mercaderia",
            "fecha_origen = :fecha_origen",
            "fecha_destino = :fecha_destino",
            "tipo_transporte = :tipo_transporte",
            "km_recorridos = :km_recorridos",
            "`CTG` = :ctg",
            "toneladas_carga = :toneladas_carga",
            "toneladas = :toneladas",
            "tarifa_real = :tarifa_real",
            "tarifa_rendida = :tarifa_rendida",
            "estado_viaje = :estado_viaje",
            "estado_facturacion = :estado_facturacion",
            "numero_factura = :numero_factura",   // <--- NUEVO
            "estado_pago = :estado_pago",
            "comentario = :comentario",
            "linea_factura = :linea_factura"
        ];

        $params = [
            'customer_id'        => $customer_id,
            'origin_id'          => $origin_id,
            'destination_id'     => $destination_id,
            'driver_id'          => $driver_id,
            'mercaderia'         => $mercaderia,
            'fecha_origen'       => $fecha_origen,
            'fecha_destino'      => $fecha_destino,
            'tipo_transporte'    => $tipo_transporte,
            'km_recorridos'      => $km_recorridos,
            'ctg'                => $ctg,
            'toneladas_carga'    => $toneladas_carga,
            'toneladas'          => $toneladas,
            'tarifa_real'        => $tarifa_real,
            'tarifa_rendida'     => $tarifa_rendida,
            'estado_viaje'       => $estado_viaje,
            'estado_facturacion' => $estado_facturacion,
            'numero_factura'     => $numero_factura,   // <--- NUEVO
            'estado_pago'        => $estado_pago,
            'comentario'         => $comentario,
            'linea_factura'      => $linea_factura,
            'id'                 => $trip_id
        ];

        if ($foto  !== '') { $sets[] = "foto  = :foto";  $params['foto']  = $foto; }
        if ($foto2 !== '') { $sets[] = "foto2 = :foto2"; $params['foto2'] = $foto2; }
        if ($foto3 !== '') { $sets[] = "foto3 = :foto3"; $params['foto3'] = $foto3; }
        if ($foto4 !== '') { $sets[] = "foto4 = :foto4"; $params['foto4'] = $foto4; }
        if ($foto5 !== '') { $sets[] = "foto5 = :foto5"; $params['foto5'] = $foto5; }

        $sql = "UPDATE trips SET ".implode(", ", $sets).", updated_at = NOW() WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        $_SESSION['success'] = "¡Viaje actualizado con éxito!";
        header('Location: trip.php?id=' . urlencode($trip_id));
        exit;

    } else {
        // --- INSERT ---
        $sql = "INSERT INTO trips (
                    customer_id,
                    origin_id, destination_id, driver_id,
                    mercaderia,
                    fecha_origen, fecha_destino,
                    tipo_transporte, km_recorridos, `CTG`,
                    toneladas_carga,
                    toneladas,
                    tarifa_real, tarifa_rendida,
                    estado_viaje, estado_facturacion, numero_factura, estado_pago,  -- <--- NUEVO numero_factura
                    foto, foto2, foto3, foto4, foto5,
                    comentario, linea_factura,
                    created_at, updated_at
                ) VALUES (
                    :customer_id,
                    :origin_id, :destination_id, :driver_id,
                    :mercaderia,
                    :fecha_origen, :fecha_destino,
                    :tipo_transporte, :km_recorridos, :ctg,
                    :toneladas_carga,
                    :toneladas,
                    :tarifa_real, :tarifa_rendida,
                    :estado_viaje, :estado_facturacion, :numero_factura, :estado_pago,  -- <--- NUEVO numero_factura
                    :foto, :foto2, :foto3, :foto4, :foto5,
                    :comentario, :linea_factura,
                    NOW(), NOW()
                )";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            'customer_id'        => $customer_id,
            'origin_id'          => $origin_id,
            'destination_id'     => $destination_id,
            'driver_id'          => $driver_id,
            'mercaderia'         => $mercaderia,
            'fecha_origen'       => $fecha_origen,
            'fecha_destino'      => $fecha_destino,
            'tipo_transporte'    => $tipo_transporte,
            'km_recorridos'      => $km_recorridos,
            'ctg'                => $ctg,
            'toneladas_carga'    => $toneladas_carga,
            'toneladas'          => $toneladas,
            'tarifa_real'        => $tarifa_real,
            'tarifa_rendida'     => $tarifa_rendida,
            'estado_viaje'       => $estado_viaje,
            'estado_facturacion' => $estado_facturacion,
            'numero_factura'     => $numero_factura, // <--- NUEVO
            'estado_pago'        => $estado_pago,
            'foto'               => $foto,
            'foto2'              => $foto2,
            'foto3'              => $foto3,
            'foto4'              => $foto4,
            'foto5'              => $foto5,
            'comentario'         => $comentario,
            'linea_factura'      => $linea_factura
        ]);

        $_SESSION['success'] = "¡Viaje creado con éxito!";
        header('Location: index.php');
        exit;
    }

} catch (Exception $e) {
    // error_log('STORE_TRIP_ERROR: ' . $e->getMessage());
    $_SESSION['error'] = "Ocurrió un error al guardar el viaje.";
    if ($trip_id) {
        header('Location: edit.php?id=' . urlencode($trip_id));
    } else {
        header('Location: index.php');
    }
    exit;
}