<?php
session_start();
require_once '../../config/db.php';
require_once '../../helpers/functions.php';
check_auth();

// Proveedores fijos que sí aparecen en el informe inferior
$proveedores_fijos = ['Aguada SRL','Axion','YPF Crespo','YPF Ruta'];


/* Header unificado */
ob_start();
include '../assets/nav2.php';
$navHtml = ob_get_clean();

// ===== Listas base =====
// SOLO chasis
$vehicles = $pdo->query("
  SELECT id, patente
  FROM vehicles
  WHERE patente IN ('AD763LR','AF711ZM','AG768OH','PQR160')
  ORDER BY FIELD(patente,'AD763LR','AF711ZM','AG768OH','PQR160')
")->fetchAll(PDO::FETCH_ASSOC);

$providers = $pdo->query("
  SELECT id, nombre, precio_actual
  FROM fuel_providers
  ORDER BY nombre
")->fetchAll(PDO::FETCH_ASSOC);


/* ===== Guardar carga (POST) ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'store') {

  // 1) Resolver proveedor
  $provider_id = null;

  // Si eligió uno del select:
  if (!empty($_POST['provider_id'])) {
    $provider_id = (int)$_POST['provider_id'];
  }

  // Si escribió uno nuevo:
  $provider_name = trim($_POST['provider_name'] ?? '');
  if ($provider_name !== '') {
    // Evitar duplicados por nombre exacto
    $chk = $pdo->prepare("SELECT id FROM fuel_providers WHERE nombre = ?");
    $chk->execute([$provider_name]);
    $pid = $chk->fetchColumn();

    if ($pid) {
      $provider_id = (int)$pid;
    } else {
      $ins = $pdo->prepare("INSERT INTO fuel_providers (nombre, precio_actual) VALUES (?, NULL)");
      $ins->execute([$provider_name]);
      $provider_id = (int)$pdo->lastInsertId();
    }
  }

  // 2) Manejar archivo (foto/pdf) — opcional
  $archivoPath = null;
  if (!empty($_FILES['archivo']['name']) && $_FILES['archivo']['error'] === UPLOAD_ERR_OK) {
    $allowed = ['jpg','jpeg','png','webp','pdf'];
    $orig    = $_FILES['archivo']['name'];
    $ext     = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
    if (in_array($ext, $allowed, true)) {
      // nombre único: fecha + usuario + random
      $base   = preg_replace('/[^a-zA-Z0-9_\-\.]/','_', pathinfo($orig, PATHINFO_FILENAME));
      $fname  = date('Ymd_His').'_'.($_SESSION['user_id'] ?? 'u').'_'.$base.'.'.$ext;

      $destDir = realpath(__DIR__.'/../uploads/fuel'); // /app/public/uploads/fuel
      if ($destDir === false) { mkdir(__DIR__.'/../uploads/fuel', 0755, true); $destDir = realpath(__DIR__.'/../uploads/fuel'); }
      $destAbs = $destDir.DIRECTORY_SEPARATOR.$fname;

      if (move_uploaded_file($_FILES['archivo']['tmp_name'], $destAbs)) {
        // Guardamos ruta relativa para servir desde web
        $archivoPath = '/app/public/uploads/fuel/'.$fname;
      }
    }
  }

  // 3) Insertar la carga
  $sql = "INSERT INTO fuel_loads
          (vehicle_id, provider_id, fecha, litros, importe, lugar, tipo, odometro_km, observaciones, archivo, created_by)
          VALUES (:vehicle_id, :provider_id, :fecha, :litros, :importe, :lugar, :tipo, :odometro_km, :observaciones, :archivo, :created_by)";
  $st = $pdo->prepare($sql);
 $st->execute([
  ':vehicle_id'    => (int)$_POST['vehicle_id'],
  ':provider_id'   => $provider_id ?: null,
  ':fecha'         => $_POST['fecha'],
  ':litros'        => (float)$_POST['litros'],
  ':importe'       => null, // <-- siempre NULL porque ya no pedimos el campo
  ':lugar'         => trim($_POST['lugar'] ?? ''),
  ':tipo'          => null, // <-- siempre NULL porque ya no pedimos el campo
  ':odometro_km'   => ($_POST['odometro_km'] !== '' ? (int)$_POST['odometro_km'] : null),
  ':observaciones' => trim($_POST['observaciones'] ?? ''),
  ':archivo'       => $archivoPath,
  ':created_by'    => $_SESSION['user_id'] ?? null,
]);


  header('Location: index.php?ok=1');
  exit;
}


/* ===== Helpers fechas/periodos ===== */
function period_bounds($offset_months = 0) {
  $first = (new DateTime('first day of this month'))->modify("{$offset_months} month")->format('Y-m-d');
  $last  = (new DateTime('last day of this month'))->modify("{$offset_months} month")->format('Y-m-d');
  return [$first, $last];
}

/* ===== Cálculo km/l compatible (sin WITH/LAG) ===== */
function fetch_metric_legacy(PDO $pdo, int $vehicle_id, string $desde, string $hasta): ?float {
  $sql = "
    SELECT 
      SUM(
        CASE
          WHEN f1.odometro_km IS NOT NULL AND (
            SELECT f2.odometro_km
            FROM fuel_loads f2
            WHERE f2.vehicle_id = f1.vehicle_id
              AND f2.fecha < f1.fecha
            ORDER BY f2.fecha DESC, f2.id DESC
            LIMIT 1
          ) IS NOT NULL
          THEN f1.odometro_km - (
            SELECT f2.odometro_km
            FROM fuel_loads f2
            WHERE f2.vehicle_id = f1.vehicle_id
              AND f2.fecha < f1.fecha
            ORDER BY f2.fecha DESC, f2.id DESC
            LIMIT 1
          )
          ELSE 0
        END
      ) AS km_sum,
      SUM(f1.litros) AS litros_sum
    FROM fuel_loads f1
    WHERE f1.vehicle_id = :vid AND f1.fecha BETWEEN :d AND :h
  ";
  $st = $pdo->prepare($sql);
  $st->execute([':vid'=>$vehicle_id, ':d'=>$desde, ':h'=>$hasta]);
  $r = $st->fetch(PDO::FETCH_ASSOC);
  $km  = (float)($r['km_sum'] ?? 0);
  $lts = (float)($r['litros_sum'] ?? 0);
  return $lts>0 ? round($km/$lts, 2) : null;
}

/* ===== Métricas por patente ===== */
$metrics = [];
foreach ($vehicles as $v) {
  $vehicle_id = (int)$v['id'];
  $minDate = $pdo->prepare("SELECT MIN(fecha) FROM fuel_loads WHERE vehicle_id = ?");
  $minDate->execute([$vehicle_id]);
  $dmin = $minDate->fetchColumn();

  [$m0d,$m0h] = period_bounds(0);   // mes actual
  [$m1d,$m1h] = period_bounds(-1);  // mes -1
  [$m2d,$m2h] = period_bounds(-2);  // mes -2

  $metrics[$vehicle_id] = [
    'patente'  => $v['patente'],
    'historic' => $dmin ? fetch_metric_legacy($pdo, $vehicle_id, $dmin, (new DateTime())->format('Y-m-d')) : null,
    'm0'       => fetch_metric_legacy($pdo, $vehicle_id, $m0d, $m0h),
    'm1'       => fetch_metric_legacy($pdo, $vehicle_id, $m1d, $m1h),
    'm2'       => fetch_metric_legacy($pdo, $vehicle_id, $m2d, $m2h),
  ];
}

/* ===== Quincenas por proveedor (actual + 3 previas) ===== */
function quin_label(DateTime $d) {
  $q = ((int)$d->format('d') <= 15) ? '1' : '2';
  return $d->format('Ym') . $q; // ej: 2025101
}
$labels = [];
$cursor = new DateTime();
for ($i=0; $i<4; $i++) {
  $labels[] = quin_label($cursor);
  if ((int)$cursor->format('d') <= 15) {
    $cursor = (new DateTime($cursor->format('Y-m-01')))->modify('-1 day'); // Q2 mes anterior
  } else {
    $cursor->setDate((int)$cursor->format('Y'), (int)$cursor->format('m'), 15); // Q1 actual
  }
}
$labels = array_values(array_unique($labels));

$in = implode(',', array_fill(0, count($labels), '?'));

$sqlProv = "
  SELECT 
    fp.id AS provider_id, 
    fp.nombre,
    CONCAT(DATE_FORMAT(fl.fecha, '%Y%m'),
           CASE WHEN DAY(fl.fecha) <= 15 THEN '1' ELSE '2' END) AS quin,
    SUM(fl.litros) AS litros,
    SUM(
      COALESCE(
        fl.importe,                            -- si viene importe, usarlo
        fl.litros * COALESCE(fp.precio_actual,0) -- si no, estimar con precio actual
      )
    ) AS monto_calc
  FROM fuel_loads fl
  JOIN fuel_providers fp ON fp.id = fl.provider_id
  WHERE CONCAT(DATE_FORMAT(fl.fecha, '%Y%m'),
               CASE WHEN DAY(fl.fecha) <= 15 THEN '1' ELSE '2' END) IN ($in)
  GROUP BY fp.id, fp.nombre, quin
  ORDER BY fp.nombre";
$st = $pdo->prepare($sqlProv);
$st->execute($labels);
$raw = $st->fetchAll(PDO::FETCH_ASSOC);


/* Pivot en PHP */
/* Pivot en PHP: SOLO proveedores fijos del informe */
$proveedores_fijos = ['Aguada SRL','Axion','YPF Crespo','YPF Ruta'];

$inNames = str_repeat('?,', count($proveedores_fijos)-1).'?';
$stFix = $pdo->prepare("
  SELECT id, nombre, precio_actual
  FROM fuel_providers
  WHERE nombre IN ($inNames)
  ORDER BY nombre
");
$stFix->execute($proveedores_fijos);
$providers_fix = $stFix->fetchAll(PDO::FETCH_ASSOC);

$providerSummary = [];
foreach ($providers_fix as $p) {
  $providerSummary[$p['id']] = [
    'nombre' => $p['nombre'],
    'precio' => $p['precio_actual'],
    'q'      => array_fill_keys($labels, ['litros'=>0.0,'monto'=>0.0]),
  ];
}

foreach ($raw as $r) {
  $pid  = (int)$r['provider_id'];
  $quin = $r['quin'];
  if (!isset($providerSummary[$pid])) continue; // solo los 4 fijos
  $providerSummary[$pid]['q'][$quin]['litros'] = (float)$r['litros'];
  $providerSummary[$pid]['q'][$quin]['monto']  = (float)$r['monto_calc']; // <- antes usábamos monto_cargado
}


?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Combustible</title>

  <!-- Bootstrap 5 + Icons (igual que trips) -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css">

  <style>
    .content-wrap { margin-top: 82px; padding-bottom: 40px; }
    body { background:#f6f7f8; font-family:"Segoe UI", Arial, sans-serif; font-size:14px; color:#222; }
    .page-card { border-radius:8px; border:1px solid #e6e9ec; background:#fff; }
    .card-metrics .badge { font-size: .85rem; }
    .table thead th { white-space: nowrap; }
  </style>
<style>
  /* Estilo botón naranja tipo Macher */
  .btn-macher {
    background-color: #f68b1e;
    color: #fff;
    font-weight: 600;
    border: none;
    transition: all 0.15s ease-in-out;
  }
  .btn-macher:hover,
  .btn-macher:focus {
    background-color: #d97706; /* tono más oscuro al pasar el mouse */
    color: #fff;
    transform: scale(1.02);
  }
</style>

</head>
<body>
  <?= $navHtml ?>

  <div class="content-wrap container">
    <div class="page-card card">
      <div class="card-body">

        <div class="d-flex justify-content-between align-items-center mb-3">
          <h5 class="mb-0 text-primary fw-bold">Combustible</h5>
        </div>

        <?php if (isset($_GET['ok'])): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            Carga guardada correctamente.
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Cerrar"></button>
          </div>
        <?php endif; ?>

        <!-- 1) Línea de carga -->
<div class="border rounded p-3 mb-4 bg-white">
  <form method="post" enctype="multipart/form-data" class="row g-2 align-items-end">
    <input type="hidden" name="action" value="store">

    <div class="col-md-2">
      <label class="form-label">Patente</label>
      <select name="vehicle_id" class="form-select" required>
        <option value="">Seleccionar…</option>
        <?php foreach ($vehicles as $v): ?>
          <option value="<?= $v['id'] ?>"><?= htmlspecialchars($v['patente']) ?></option>
        <?php endforeach; ?>
      </select>
    </div>

    <div class="col-md-2">
      <label class="form-label">Fecha</label>
      <input type="date" name="fecha" class="form-control" required>
    </div>

    <div class="col-md-2">
      <label class="form-label">Proveedor</label>
      <select name="provider_id" id="provider_id" class="form-select">
        <option value="">— Seleccionar —</option>
        <?php foreach ($providers as $p): ?>
          <option value="<?= $p['id'] ?>"><?= htmlspecialchars($p['nombre']) ?></option>
        <?php endforeach; ?>
        <option value="__otro__">— Escribir otro… —</option>
      </select>
    </div>

    <div class="col-md-2">
      <label class="form-label">Nuevo proveedor</label>
      <input type="text" name="provider_name" id="provider_name" class="form-control" placeholder="Nombre proveedor" disabled>
    </div>

    <div class="col-md-2">
      <label class="form-label">Litros</label>
      <input type="number" step="0.001" name="litros" class="form-control" required>
    </div>

    <div class="col-md-2">
      <label class="form-label">Odómetro (km)</label>
      <input type="number" name="odometro_km" class="form-control">
    </div>

    <div class="col-md-2">
      <label class="form-label">Lugar</label>
      <input type="text" name="lugar" class="form-control" placeholder="Estación / Localidad">
    </div>

    <div class="col-md-3">
      <label class="form-label">Observaciones</label>
      <input type="text" name="observaciones" class="form-control">
    </div>

    <div class="col-md-3">
      <label class="form-label">Adjunto (jpg/png/webp/pdf)</label>
      <input type="file" name="archivo" class="form-control" accept=".jpg,.jpeg,.png,.webp,.pdf">
    </div>

    <div class="col-md-2 d-grid">
      <button class="btn btn-primary">Cargar</button>
    </div>
  </form>
</div>


<!-- ... (Litros, Importe, Odómetro) ... -->

        <!-- 2) Informes por patente -->
        <div class="card-metrics mb-4">
          <div class="d-flex justify-content-between align-items-center mb-2">
            <h6 class="mb-0 fw-bold">Informes por patente (km/l)</h6>
            
          </div>

 <div class="row g-3">
  <?php foreach ($metrics as $vid => $m): ?>
    <div class="col-sm-6 col-md-4 col-lg-3">
      <div class="border rounded p-3 bg-white h-100 text-center">
        <a href="loads.php?vehicle_id=<?= $vid ?>" 
           class="btn btn-macher w-100 mb-3">
          <?= htmlspecialchars($m['patente']) ?>
        </a>
        <div class="d-flex flex-column gap-1">
          <span class="badge text-bg-light w-100 d-flex justify-content-between">
            <span>Histórico</span><span><?= $m['historic'] ?? '—' ?></span>
          </span>
          <span class="badge text-bg-light w-100 d-flex justify-content-between">
            <span>Mes actual</span><span><?= $m['m0'] ?? '—' ?></span>
          </span>
          <span class="badge text-bg-light w-100 d-flex justify-content-between">
            <span>Mes -1</span><span><?= $m['m1'] ?? '—' ?></span>
          </span>
          <span class="badge text-bg-light w-100 d-flex justify-content-between">
            <span>Mes -2</span><span><?= $m['m2'] ?? '—' ?></span>
          </span>
        </div>
      </div>
    </div>
  <?php endforeach; ?>
</div>


        <!-- 3) Consumo por proveedor (quincenas) -->
        <div id="proveedores" class="border rounded p-3 bg-white">
          <div class="d-flex justify-content-between align-items-center mb-2">
            <h6 class="mb-0 fw-bold">Consumo por proveedor</h6>
          </div>

          <form action="update_prices.php" method="post">
            <div class="table-responsive">
              <table class="table table-sm table-striped align-middle">
                <thead class="table-light">
                  <tr>
                    <th>Proveedor</th>
                    <th>Precio actual ($/lt)</th>
                    <?php foreach ($labels as $lab): ?>
                      <th class="text-center"><?= htmlspecialchars($lab) ?></th>
                    <?php endforeach; ?>
                  </tr>
                </thead>
                <tbody>
                  <?php foreach ($providerSummary as $pid => $row): ?>
                    <tr>
                      <td><?= htmlspecialchars($row['nombre']) ?></td>
                      <td style="max-width:140px">
                        <input type="number" step="0.001" class="form-control form-control-sm"
                               name="precio[<?= $pid ?>]" value="<?= htmlspecialchars((string)$row['precio']) ?>">
                      </td>
                      <?php foreach ($labels as $lab):
                        $lit = $row['q'][$lab]['litros'];
                        $monto = $row['q'][$lab]['monto'];
                      ?>
                        <td class="text-center">
                          <div><strong><?= number_format($lit, 2, ',', '.') ?></strong> l</div>
                          <div class="text-muted">$ <?= number_format($monto, 2, ',', '.') ?></div>
                        </td>
                      <?php endforeach; ?>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>

            <div class="mt-2">
              <button class="btn btn-primary btn-sm">Guardar precios</button>
            </div>
          </form>

          <p class="text-muted small mt-2 mb-0">
            Nota: si una carga no tiene “importe”, el monto puede calcularse con el precio actual del proveedor (litros × precio).
          </p>
        </div>

      </div>
    </div>
  </div>

  <script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
  (function(){
    const sel = document.getElementById('provider_id');
    const txt = document.getElementById('provider_name');
    if (!sel || !txt) return;

    function toggle() {
      if (sel.value === '__otro__') {
        txt.disabled = false;
        txt.required = true;
        txt.focus();
      } else {
        txt.value = '';
        txt.disabled = true;
        txt.required = false;
      }
    }
    sel.addEventListener('change', toggle);
    toggle();
  })();
</script>

</body>
</html>