<?php
session_start();
require_once '../../config/db.php';
require_once '../../helpers/functions.php';
check_auth();

/* ====== DOMPDF ======
   Requiere Dompdf instalado (composer).
   Si no lo tenés, abajo te digo cómo verificarlo.
*/
$autoload1 = __DIR__ . '/../../vendor/autoload.php';
$autoload2 = __DIR__ . '/../vendor/autoload.php';
if (file_exists($autoload1)) {
    require_once $autoload1;
} elseif (file_exists($autoload2)) {
    require_once $autoload2;
} else {
    die('No se encontró autoload.php de Composer. Necesitás Dompdf instalado.');
}

use Dompdf\Dompdf;
use Dompdf\Options;

/* Helper para texto */
function h($v) {
    return htmlspecialchars((string)($v ?? ''), ENT_QUOTES, 'UTF-8');
}

/* ===========================
   Params: id + mes/anio
   =========================== */
$driver_id = $_GET['id'] ?? null;
if (!$driver_id || !ctype_digit((string)$driver_id)) die('Chofer inválido.');
$driver_id = (int)$driver_id;

$mes  = $_GET['mes']  ?? date('m');
$anio = $_GET['anio'] ?? date('Y');

$mes  = str_pad(preg_replace('/\D/', '', (string)$mes), 2, '0', STR_PAD_LEFT);
$anio = preg_replace('/\D/', '', (string)$anio);

if ((int)$mes < 1 || (int)$mes > 12) $mes = date('m');
if ((int)$anio < 2000 || (int)$anio > 2100) $anio = date('Y');

$periodo_inicio = $anio . '-' . $mes . '-01';
$periodo_fin    = date('Y-m-d', strtotime($periodo_inicio . ' +1 month')); // exclusivo

$meses = [
  '01'=>'Enero','02'=>'Febrero','03'=>'Marzo','04'=>'Abril','05'=>'Mayo','06'=>'Junio',
  '07'=>'Julio','08'=>'Agosto','09'=>'Septiembre','10'=>'Octubre','11'=>'Noviembre','12'=>'Diciembre'
];

/* ===========================
   Chofer
   =========================== */
$stmt = $pdo->prepare("SELECT * FROM drivers WHERE id = :id LIMIT 1");
$stmt->execute(['id' => $driver_id]);
$driver = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$driver) die('Chofer no encontrado.');

/* ===========================
   Saldo anterior (movimientos previos)
   =========================== */
$stmt = $pdo->prepare("
    SELECT
      COALESCE(SUM(haber),0) AS sum_haber,
      COALESCE(SUM(debe),0)  AS sum_debe
    FROM driver_movements
    WHERE driver_id = :driver_id
      AND estado = 1
      AND fecha < :inicio
");
$stmt->execute([
    'driver_id' => $driver_id,
    'inicio'    => $periodo_inicio
]);
$prev = $stmt->fetch(PDO::FETCH_ASSOC);
$saldo_anterior_mov = (float)$prev['sum_haber'] - (float)$prev['sum_debe'];

/* ===========================
   Movimientos del mes
   =========================== */
$stmt = $pdo->prepare("
    SELECT *
    FROM driver_movements
    WHERE driver_id = :driver_id
      AND estado = 1
      AND fecha >= :inicio
      AND fecha <  :fin
    ORDER BY fecha ASC, id ASC
");
$stmt->execute([
    'driver_id' => $driver_id,
    'inicio'    => $periodo_inicio,
    'fin'       => $periodo_fin
]);
$mov_mes = $stmt->fetchAll(PDO::FETCH_ASSOC);

$tot_debe_mes  = 0.0;
$tot_haber_mes = 0.0;
foreach ($mov_mes as $m) {
    $tot_debe_mes  += (float)$m['debe'];
    $tot_haber_mes += (float)$m['haber'];
}

/* ===========================
   Viajes del mes (Realizado) + 16%
   =========================== */
$stmt = $pdo->prepare("
    SELECT
      id,
      toneladas,
      tarifa_rendida,
      fecha_origen,
      fecha_destino,
      linea_factura,
      comentario
    FROM trips
    WHERE driver_id = :driver_id
      AND estado_viaje = 'Realizado'
      AND (
            (fecha_destino IS NOT NULL AND fecha_destino >= :inicio AND fecha_destino < :fin)
         OR (fecha_destino IS NULL AND fecha_origen >= :inicio AND fecha_origen < :fin)
      )
    ORDER BY COALESCE(fecha_destino, fecha_origen) ASC, id ASC
");
$stmt->execute([
    'driver_id' => $driver_id,
    'inicio'    => $periodo_inicio,
    'fin'       => $periodo_fin
]);
$viajes_mes = $stmt->fetchAll(PDO::FETCH_ASSOC);

$tot_tn_mes   = 0.0;
$tot_base_mes = 0.0;
$tot_16_mes   = 0.0;

foreach ($viajes_mes as $v) {
    $tn   = (float)($v['toneladas'] ?? 0);
    $tar  = (float)($v['tarifa_rendida'] ?? 0);
    $base = $tn * $tar;
    $liq16 = $base * 0.16;

    $tot_tn_mes   += $tn;
    $tot_base_mes += $base;
    $tot_16_mes   += $liq16;
}

/* ===========================
   Lógica de liquidación (la que ya venís usando)
   saldo_final = saldo_anterior + total_16% + (haber_mes - debe_mes)
   =========================== */
$saldo_final = $saldo_anterior_mov + $tot_16_mes + ($tot_haber_mes - $tot_debe_mes);

/* ===========================
   Armar HTML del PDF
   =========================== */
$nombreChofer = trim(($driver['apellido'] ?? '') . ' ' . ($driver['nombre'] ?? ''));
$periodoTxt = ($meses[$mes] ?? $mes) . ' ' . $anio;

$saldo_anterior_fmt = number_format($saldo_anterior_mov, 2, ',', '.');
$tot_16_fmt         = number_format($tot_16_mes, 2, ',', '.');
$tot_debe_fmt       = number_format($tot_debe_mes, 2, ',', '.');
$tot_haber_fmt      = number_format($tot_haber_mes, 2, ',', '.');
$saldo_final_fmt    = number_format($saldo_final, 2, ',', '.');

$html = '
<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="utf-8">
<style>
  body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; color:#111; }
  .title { font-size: 16px; font-weight: 700; margin-bottom: 6px; }
  .sub { font-size: 12px; color:#333; margin-bottom: 12px; }
  .box { border:1px solid #ddd; padding:10px; border-radius:6px; margin-bottom: 10px; }
  .kpi { width:100%; border-collapse: collapse; }
  .kpi td { padding:6px 8px; vertical-align: top; }
  .kpi .lbl { color:#555; width: 200px; }
  .kpi .val { font-weight:700; }
  table { width:100%; border-collapse: collapse; }
  th, td { border:1px solid #e1e1e1; padding:6px 6px; }
  th { background:#f3f4f6; text-align:left; }
  .right { text-align:right; }
  .center { text-align:center; }
  .muted { color:#666; }
  .totrow td { font-weight:700; background:#fafafa; }
  .footerbox { border:2px solid #111; padding:10px; margin-top:10px; }
  .big { font-size: 14px; font-weight: 800; }
</style>
</head>
<body>

<div class="title">Liquidación - ' . h($periodoTxt) . '</div>
<div class="sub">Chofer: <strong>' . h($nombreChofer) . '</strong> | ID: ' . h($driver_id) . '</div>

<div class="box">
  <table class="kpi">
    <tr>
      <td class="lbl">Saldo mes anterior</td>
      <td class="val">$ ' . $saldo_anterior_fmt . '</td>
      <td class="lbl">Total viajes (16%)</td>
      <td class="val">$ ' . $tot_16_fmt . '</td>
    </tr>
    <tr>
      <td class="lbl">Movimientos del mes (Debe)</td>
      <td class="val">$ ' . $tot_debe_fmt . '</td>
      <td class="lbl">Movimientos del mes (Haber)</td>
      <td class="val">$ ' . $tot_haber_fmt . '</td>
    </tr>
  </table>
</div>

<div class="box">
  <div style="font-weight:700; margin-bottom:6px;">Viajes realizados</div>
  <table>
    <thead>
      <tr>
        <th style="width:85px;">Fecha</th>
        <th>Resumen del viaje</th>
        <th class="right" style="width:80px;">Tn</th>
        <th class="right" style="width:95px;">Tarifa</th>
        <th class="right" style="width:115px;">Base</th>
        <th class="right" style="width:95px;">16%</th>
      </tr>
    </thead>
    <tbody>
';

if (empty($viajes_mes)) {
    $html .= '<tr><td colspan="6" class="center muted">Sin viajes realizados en este período.</td></tr>';
} else {
    foreach ($viajes_mes as $v) {
        $tn   = (float)($v['toneladas'] ?? 0);
        $tar  = (float)($v['tarifa_rendida'] ?? 0);
        $base = $tn * $tar;
        $liq16 = $base * 0.16;

        $fecha = $v['fecha_destino'] ?: $v['fecha_origen'];

        $resumen = trim((string)($v['linea_factura'] ?? ''));
        if ($resumen === '') $resumen = trim((string)($v['comentario'] ?? ''));
        if ($resumen === '') $resumen = 'Viaje #' . (int)$v['id'];

        $html .= '
        <tr>
          <td>' . h($fecha) . '</td>
          <td>' . h($resumen) . '</td>
          <td class="right">' . number_format($tn, 2, ',', '.') . '</td>
          <td class="right">$ ' . number_format($tar, 2, ',', '.') . '</td>
          <td class="right">$ ' . number_format($base, 2, ',', '.') . '</td>
          <td class="right"><strong>$ ' . number_format($liq16, 2, ',', '.') . '</strong></td>
        </tr>';
    }

    $html .= '
      <tr class="totrow">
        <td colspan="2" class="right">Totales</td>
        <td class="right">' . number_format($tot_tn_mes, 2, ',', '.') . '</td>
        <td></td>
        <td class="right">$ ' . number_format($tot_base_mes, 2, ',', '.') . '</td>
        <td class="right">$ ' . number_format($tot_16_mes, 2, ',', '.') . '</td>
      </tr>';
}

$html .= '
    </tbody>
  </table>
</div>

<div class="box">
  <div style="font-weight:700; margin-bottom:6px;">Movimientos del mes</div>
  <table>
    <thead>
      <tr>
        <th style="width:85px;">Fecha</th>
        <th style="width:90px;">Tipo</th>
        <th>Descripción</th>
        <th style="width:110px;">Medio de pago</th>
        <th class="right" style="width:95px;">Debe</th>
        <th class="right" style="width:95px;">Haber</th>
      </tr>
    </thead>
    <tbody>
';

if (empty($mov_mes)) {
    $html .= '<tr><td colspan="6" class="center muted">Sin movimientos cargados en este período.</td></tr>';
} else {
    foreach ($mov_mes as $m) {
        $html .= '
        <tr>
          <td>' . h($m['fecha']) . '</td>
          <td>' . h($m['tipo']) . '</td>
          <td>' . h($m['descripcion']) . '</td>
          <td>' . h($m['medio_pago']) . '</td>
          <td class="right">' . (((float)$m['debe'] != 0.0) ? '$ ' . number_format((float)$m['debe'], 2, ',', '.') : '-') . '</td>
          <td class="right">' . (((float)$m['haber'] != 0.0) ? '$ ' . number_format((float)$m['haber'], 2, ',', '.') : '-') . '</td>
        </tr>';
    }

    $html .= '
      <tr class="totrow">
        <td colspan="4" class="right">Totales</td>
        <td class="right">$ ' . $tot_debe_fmt . '</td>
        <td class="right">$ ' . $tot_haber_fmt . '</td>
      </tr>';
}

$html .= '
    </tbody>
  </table>
</div>

<div class="footerbox">
  <div class="big">TOTAL A COBRAR / SALDO DEL MES</div>
  <div style="margin-top:6px;">
    <span class="muted">Fórmula:</span>
    <strong>Saldo mes anterior</strong> + <strong>Total viajes (16%)</strong> + (<strong>Haber</strong> - <strong>Debe</strong>)
  </div>
  <div style="margin-top:8px; font-size:18px; font-weight:900;">
    $ ' . $saldo_final_fmt . '
  </div>
</div>

</body>
</html>
';

/* ===========================
   Generar PDF (A4 horizontal)
   =========================== */
$options = new Options();
$options->set('isRemoteEnabled', false);

$dompdf = new Dompdf($options);
$dompdf->setPaper('A4', 'landscape');
$dompdf->loadHtml($html, 'UTF-8');
$dompdf->render();

/* Nombre del archivo */
$mesNombre = $meses[$mes] ?? $mes;
$fname = 'Liquidacion ' . $mesNombre . ' ' . $anio . ' - ' . $nombreChofer . '.pdf';
$fname = preg_replace('/[^\w\s\-\.\(\)]/u', '', $fname);

header('Content-Type: application/pdf');
header('Content-Disposition: attachment; filename="' . $fname . '"');
echo $dompdf->output();
exit;