<?php
session_start();
require_once '../../config/db.php';
require_once '../../helpers/functions.php';
check_auth();

/* ===========================
   DEBUG LOG SIMPLE (cc_error.log)
   =========================== */
error_reporting(E_ALL);
ini_set('display_errors', '0'); // no mostrar en pantalla
ini_set('log_errors', '1');     // en 1 para que pueda loguear

$__cc_log_file = __DIR__ . '/cc_error.log';

function cc_log($msg) {
    global $__cc_log_file;
    $line = '[' . date('Y-m-d H:i:s') . '] ' . $msg . PHP_EOL;
    @file_put_contents($__cc_log_file, $line, FILE_APPEND);
}

set_exception_handler(function($e){
    cc_log("EXCEPTION: " . $e->getMessage() . " | File: " . $e->getFile() . ":" . $e->getLine());
    http_response_code(500);
    exit;
});

set_error_handler(function($severity, $message, $file, $line){
    cc_log("ERROR: $message | File: $file:$line");
    return false;
});

/* Helper para texto */
function h($v) {
    return htmlspecialchars((string)($v ?? ''), ENT_QUOTES, 'UTF-8');
}

/* ===========================
   1) Entrada: id del chofer + periodo (mes/año)
   =========================== */
$driver_id = $_GET['id'] ?? null;
if (!$driver_id || !ctype_digit((string)$driver_id)) {
    header('Location: index.php');
    exit;
}
$driver_id = (int)$driver_id;

// Periodo seleccionado (mantiene el seleccionado en pantalla)
$mes  = $_GET['mes']  ?? date('m');
$anio = $_GET['anio'] ?? date('Y');

$mes  = str_pad(preg_replace('/\D/', '', (string)$mes), 2, '0', STR_PAD_LEFT);
$anio = preg_replace('/\D/', '', (string)$anio);

if ((int)$mes < 1 || (int)$mes > 12) $mes = date('m');
if ((int)$anio < 2000 || (int)$anio > 2100) $anio = date('Y');

$periodo_inicio = $anio . '-' . $mes . '-01';
$periodo_fin    = date('Y-m-d', strtotime($periodo_inicio . ' +1 month')); // exclusivo

/* Buscar chofer */
$stmt = $pdo->prepare("SELECT * FROM drivers WHERE id = :id LIMIT 1");
$stmt->execute(['id' => $driver_id]);
$driver = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$driver) {
    $_SESSION['error'] = "Chofer no encontrado.";
    header('Location: index.php');
    exit;
}

/* ===========================
   2) Eliminar movimiento
   =========================== */
if (isset($_GET['delete']) && ctype_digit($_GET['delete'])) {
    $mov_id = (int) $_GET['delete'];

    $stmt = $pdo->prepare("
        SELECT adjunto
        FROM driver_movements
        WHERE id = :id AND driver_id = :driver_id
    ");
    $stmt->execute([
        'id'        => $mov_id,
        'driver_id' => $driver_id
    ]);
    $mov = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($mov) {
        if (!empty($mov['adjunto'])) {
            $filePath = __DIR__ . '/../uploads/' . $mov['adjunto'];
            if (is_file($filePath)) {
                @unlink($filePath);
            }
        }

        $del = $pdo->prepare("
            DELETE FROM driver_movements
            WHERE id = :id AND driver_id = :driver_id
        ");
        $del->execute([
            'id'        => $mov_id,
            'driver_id' => $driver_id
        ]);

        $_SESSION['success'] = "Movimiento eliminado correctamente.";
    } else {
        $_SESSION['error'] = "Movimiento no encontrado.";
    }

    header('Location: cc.php?id=' . urlencode($driver_id) . '&mes=' . urlencode($mes) . '&anio=' . urlencode($anio));
    exit;
}

/* ===========================
   3) Alta de nuevo movimiento
   =========================== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $fecha       = $_POST['fecha'] ?? date('Y-m-d');
    $tipo        = $_POST['tipo'] ?? 'ajuste';
    $descripcion = $_POST['descripcion'] ?? '';
    $medio_pago  = $_POST['medio_pago'] ?? '';

    // Normalizar números: "1.234,56" -> "1234.56"
    $debe_raw  = str_replace(['.', ','], ['', '.'], $_POST['debe']  ?? '0');
    $haber_raw = str_replace(['.', ','], ['', '.'], $_POST['haber'] ?? '0');

    $debe  = (float)$debe_raw;
    $haber = (float)$haber_raw;

    // Manejo del archivo
    $adjuntoDb = null;
    if (!empty($_FILES['adjunto']['name']) && ($_FILES['adjunto']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK) {
        $tmpName = $_FILES['adjunto']['tmp_name'];
        $orig    = $_FILES['adjunto']['name'];

        $ext = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
        $allowed = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'pdf'];

        if (in_array($ext, $allowed, true)) {
            $subdir    = 'driver_cc';
            $uploadDir = __DIR__ . '/../uploads/' . $subdir . '/';

            if (!is_dir($uploadDir)) {
                @mkdir($uploadDir, 0775, true);
            }

            $newName = 'cc_' . $driver_id . '_' . date('Ymd_His') . '_' . bin2hex(random_bytes(3)) . '.' . $ext;
            $dest    = $uploadDir . $newName;

            if (move_uploaded_file($tmpName, $dest)) {
                $adjuntoDb = $subdir . '/' . $newName; // relativo a /uploads
            }
        }
    }

    $sql = "INSERT INTO driver_movements
            (driver_id, fecha, tipo, trip_id, descripcion, debe, haber, adjunto, medio_pago, origen, user_id, estado, created_at)
            VALUES
            (:driver_id, :fecha, :tipo, :trip_id, :descripcion, :debe, :haber, :adjunto, :medio_pago, :origen, :user_id, :estado, NOW())";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        'driver_id'   => $driver_id,
        'fecha'       => $fecha,
        'tipo'        => $tipo,
        'trip_id'     => null,
        'descripcion' => $descripcion,
        'debe'        => $debe,
        'haber'       => $haber,
        'adjunto'     => $adjuntoDb,
        'medio_pago'  => $medio_pago,
        'origen'      => 'manual',
        'user_id'     => $_SESSION['user_id'] ?? null,
        'estado'      => 1
    ]);

    $_SESSION['success'] = "Movimiento cargado correctamente.";
    header('Location: cc.php?id=' . urlencode($driver_id) . '&mes=' . urlencode($mes) . '&anio=' . urlencode($anio));
    exit;
}

/* ===========================
   4) Movimientos del mes + saldo anterior
   =========================== */
// saldo anterior: suma de (haber-debe) antes del periodo
$stmt = $pdo->prepare("
    SELECT
      COALESCE(SUM(haber),0) AS sum_haber,
      COALESCE(SUM(debe),0)  AS sum_debe
    FROM driver_movements
    WHERE driver_id = :driver_id
      AND estado = 1
      AND fecha < :inicio
");
$stmt->execute([
    'driver_id' => $driver_id,
    'inicio'    => $periodo_inicio
]);
$prev = $stmt->fetch(PDO::FETCH_ASSOC);
$saldo_anterior_mov = (float)$prev['sum_haber'] - (float)$prev['sum_debe'];

// movimientos del mes
$stmt = $pdo->prepare("
    SELECT *
    FROM driver_movements
    WHERE driver_id = :driver_id
      AND estado = 1
      AND fecha >= :inicio
      AND fecha <  :fin
    ORDER BY fecha ASC, id ASC
");
$stmt->execute([
    'driver_id' => $driver_id,
    'inicio'    => $periodo_inicio,
    'fin'       => $periodo_fin
]);
$mov_mes = $stmt->fetchAll(PDO::FETCH_ASSOC);

/* Totales movimientos del mes */
$tot_debe_mes  = 0.0;
$tot_haber_mes = 0.0;
foreach ($mov_mes as $m) {
    $tot_debe_mes  += (float)$m['debe'];
    $tot_haber_mes += (float)$m['haber'];
}

/* ===========================
   5) Viajes del mes + cálculo 16%
   =========================== */
$stmt = $pdo->prepare("
    SELECT
      id,
      mercaderia,
      toneladas,
      tarifa_rendida,
      fecha_origen,
      fecha_destino,
      comentario,
      linea_factura,
      CTG
    FROM trips
    WHERE driver_id = :driver_id
      AND estado_viaje = 'Realizado'
      AND (
            (fecha_destino IS NOT NULL AND fecha_destino >= :inicio AND fecha_destino < :fin)
         OR (fecha_destino IS NULL AND fecha_origen >= :inicio AND fecha_origen < :fin)
      )
    ORDER BY COALESCE(fecha_destino, fecha_origen) ASC, id ASC
");
$stmt->execute([
    'driver_id' => $driver_id,
    'inicio'    => $periodo_inicio,
    'fin'       => $periodo_fin
]);
$viajes_mes = $stmt->fetchAll(PDO::FETCH_ASSOC);

$tot_tn_mes = 0.0;
$tot_base_mes = 0.0;
$tot_16_mes = 0.0;

foreach ($viajes_mes as $v) {
    $tn = (float)($v['toneladas'] ?? 0);
    $tar = (float)($v['tarifa_rendida'] ?? 0);
    $base = $tn * $tar;
    $liq16 = $base * 0.16;

    $tot_tn_mes   += $tn;
    $tot_base_mes += $base;
    $tot_16_mes   += $liq16;
}

/* ===========================
   6) SALDO FINAL del mes
   =========================== */
$saldo_final = $saldo_anterior_mov + $tot_16_mes + ($tot_haber_mes - $tot_debe_mes);

$saldo_final_fmt = number_format($saldo_final, 2, ',', '.');
$saldo_anterior_fmt = number_format($saldo_anterior_mov, 2, ',', '.');
$tot_16_fmt = number_format($tot_16_mes, 2, ',', '.');
$tot_debe_fmt = number_format($tot_debe_mes, 2, ',', '.');
$tot_haber_fmt = number_format($tot_haber_mes, 2, ',', '.');

/* Mensajes */
$success = $_SESSION['success'] ?? null;
$error   = $_SESSION['error'] ?? null;
unset($_SESSION['success'], $_SESSION['error']);

/* Meses para selector */
$meses = [
  '01'=>'Enero','02'=>'Febrero','03'=>'Marzo','04'=>'Abril','05'=>'Mayo','06'=>'Junio',
  '07'=>'Julio','08'=>'Agosto','09'=>'Septiembre','10'=>'Octubre','11'=>'Noviembre','12'=>'Diciembre'
];
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Cuenta Corriente - <?= h($driver['nombre'] . ' ' . $driver['apellido']) ?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1">

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css">

  <style>
    .content-wrap { margin-top: 82px; padding-bottom: 40px; }
    body { background:#f6f7f8; font-family:"Segoe UI", Arial, sans-serif; font-size:14px; color:#222; }

    /* look tipo Combustible */
    .page-card { border-radius:8px; border:1px solid #e6e9ec; background:#fff; }
    .block { border:1px solid #e6e9ec; border-radius:8px; background:#fff; padding:14px; }

    .saldo-a-favor { color:#059669; font-weight:600; }
    .saldo-en-contra { color:#dc2626; font-weight:600; }

    .table td, .table th { vertical-align: middle; }
    .table thead th { white-space: nowrap; }

    /* Descripción en 2 líneas máximo */
    .clamp-2 {
      display: -webkit-box;
      -webkit-line-clamp: 2;
      -webkit-box-orient: vertical;
      overflow: hidden;
      text-overflow: ellipsis;
      white-space: normal;
      line-height: 1.2;
      max-height: calc(1.2em * 2);
    }

    /* ====== TABLA VIAJES: columnas rígidas sin wrap ====== */
    .viajes-table { table-layout: auto; }
    .viajes-table th.col-fecha,
    .viajes-table td.col-fecha,
    .viajes-table th.col-tn,
    .viajes-table td.col-tn,
    .viajes-table th.col-tarifa,
    .viajes-table td.col-tarifa,
    .viajes-table th.col-base,
    .viajes-table td.col-base,
    .viajes-table th.col-16,
    .viajes-table td.col-16{
      white-space: nowrap;
      width: 1%;
    }

    /* Resumen: una sola línea y con ... (es el que se “adapta”) */
    .viajes-table th.col-resumen,
    .viajes-table td.col-resumen{
      width: 100%;
      max-width: 0; /* clave para que el ellipsis funcione en tablas */
    }
    .ellipsis-1{
      display: block;
      white-space: nowrap;
      overflow: hidden;
      text-overflow: ellipsis;
    }

    .btn { transition: filter .15s ease; }
    .btn:hover { filter:brightness(0.92); }

    /* Ancho intermedio fijo */
    .cc-wrap{
      max-width: 1500px;
      margin-left: auto;
      margin-right: auto;
      padding-left: 16px;
      padding-right: 16px;
    }
  </style>
</head>
<body>
<?php include "../assets/nav2.php"; ?>

<main class="content-wrap cc-wrap">

  <!-- CABECERA -->
  <div class="page-card card shadow-sm mb-3">
    <div class="card-body py-3">
      <div class="d-flex justify-content-between align-items-center">
        <div>
          <h5 class="mb-0 text-primary fw-bold">
            Cuenta Corriente - <?= h($driver['nombre'] . ' ' . $driver['apellido']) ?>
          </h5>
          <div class="text-muted" style="font-size:0.9rem;">
            ID Chofer: <strong><?= h($driver_id) ?></strong> — Período: <strong><?= h($mes) ?>/<?= h($anio) ?></strong>
          </div>
        </div>

        <div class="text-end">
          <?php if ($saldo_final >= 0): ?>
            <div class="saldo-a-favor">Saldo final del mes: $ <?= $saldo_final_fmt ?></div>
          <?php else: ?>
            <div class="saldo-en-contra">Saldo final del mes: $ <?= $saldo_final_fmt ?></div>
          <?php endif; ?>

          <button
            type="button"
            onclick="window.open(
              'cc_print.php?id=<?= h($driver_id) ?>&mes=<?= h($mes) ?>&anio=<?= h($anio) ?>',
              '_blank'
            )"
            class="btn btn-warning btn-sm text-white"
            style="background-color:#f68b1e;border:none;"
          >
            <i class="bi bi-printer"></i> Imprimir liquidación
          </button>

          <a href="index.php" class="btn btn-dark btn-sm text-white mt-2">
            <i class="bi bi-arrow-left"></i> Volver
          </a>
        </div>
      </div>

      <?php if ($success): ?>
        <div class="alert alert-success py-2 mt-3 mb-0"><?= h($success) ?></div>
      <?php endif; ?>
      <?php if ($error): ?>
        <div class="alert alert-danger py-2 mt-3 mb-0"><?= h($error) ?></div>
      <?php endif; ?>
    </div>
  </div>

  <!-- BLOQUES APILADOS -->
  <div class="row g-3">
    <div class="col-12">

      <!-- BLOQUE 1: FILTRO + VIAJES -->
      <div class="block mb-3">
        <!-- Selector de mes/año -->
        <form class="row g-2 align-items-end mb-3" method="GET" action="cc.php">
          <input type="hidden" name="id" value="<?= h($driver_id) ?>">

          <div class="col-6 col-md-3">
            <label class="form-label mb-1">Mes</label>
            <select name="mes" class="form-select form-select-sm">
              <?php foreach ($meses as $k=>$label): ?>
                <option value="<?= h($k) ?>" <?= ((string)$k === (string)$mes ? 'selected' : '') ?>>
                  <?= h($label) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="col-6 col-md-3">
            <label class="form-label mb-1">Año</label>
            <select name="anio" class="form-select form-select-sm">
              <?php for($y = (int)date('Y')-3; $y <= (int)date('Y')+1; $y++): ?>
                <option value="<?= $y ?>" <?= ((string)$y === (string)$anio ? 'selected' : '') ?>><?= $y ?></option>
              <?php endfor; ?>
            </select>
          </div>

          <div class="col-12 col-md-2 d-grid">
            <button class="btn btn-primary btn-sm text-white" type="submit">
              <i class="bi bi-search"></i> Ver
            </button>
          </div>

          <div class="col-12 col-md-4 text-muted" style="font-size:0.85rem;">
            Saldo mes anterior (movimientos): <strong>$ <?= $saldo_anterior_fmt ?></strong><br>
            Total viajes (16%): <strong>$ <?= $tot_16_fmt ?></strong> — Movimientos mes: Haber <strong>$ <?= $tot_haber_fmt ?></strong> / Debe <strong>$ <?= $tot_debe_fmt ?></strong>
          </div>
        </form>

        <h6 class="fw-bold mb-2">Viajes realizados (<?= h($mes) ?>/<?= h($anio) ?>)</h6>

        <div class="table-responsive">
          <table class="table table-sm table-striped table-hover align-middle mb-2 viajes-table">
            <thead class="table-light">
              <tr>
                <th class="col-fecha">Fecha</th>
                <th class="col-resumen">Resumen del viaje</th>
                <th class="text-end col-tn">Tn</th>
                <th class="text-end col-tarifa">Tarifa rendida</th>
                <th class="text-end col-base">Base (Tn x Tarifa)</th>
                <th class="text-end col-16">16%</th>
              </tr>
            </thead>
            <tbody>
              <?php if (empty($viajes_mes)): ?>
                <tr>
                  <td colspan="6" class="text-center text-muted">Sin viajes realizados en este período.</td>
                </tr>
              <?php else: ?>
                <?php foreach ($viajes_mes as $v):
                  $tn = (float)($v['toneladas'] ?? 0);
                  $tar = (float)($v['tarifa_rendida'] ?? 0);
                  $base = $tn * $tar;
                  $liq16 = $base * 0.16;

                  $fecha = $v['fecha_destino'] ?: $v['fecha_origen'];

                  // RESUMEN DEL VIAJE: prioriza linea_factura
                  $resumen = trim((string)($v['linea_factura'] ?? ''));
                  if ($resumen === '') $resumen = trim((string)($v['comentario'] ?? ''));
                  if ($resumen === '') $resumen = 'Viaje #' . (int)$v['id'];
                ?>
                  <tr>
                    <td class="col-fecha"><?= h($fecha) ?></td>
                    <td class="col-resumen"><span class="ellipsis-1"><?= h($resumen) ?></span></td>
                    <td class="text-end col-tn"><?= number_format($tn, 2, ',', '.') ?></td>
                    <td class="text-end col-tarifa">$ <?= number_format($tar, 2, ',', '.') ?></td>
                    <td class="text-end col-base">$ <?= number_format($base, 2, ',', '.') ?></td>
                    <td class="text-end fw-semibold col-16">$ <?= number_format($liq16, 2, ',', '.') ?></td>
                  </tr>
                <?php endforeach; ?>

                <tr class="table-light">
                  <th colspan="2" class="text-end">Totales del mes</th>
                  <th class="text-end col-tn"><?= number_format($tot_tn_mes, 2, ',', '.') ?></th>
                  <th class="col-tarifa"></th>
                  <th class="text-end col-base">$ <?= number_format($tot_base_mes, 2, ',', '.') ?></th>
                  <th class="text-end col-16">$ <?= number_format($tot_16_mes, 2, ',', '.') ?></th>
                </tr>
              <?php endif; ?>
            </tbody>
          </table>
        </div>

        <div class="text-muted" style="font-size:0.85rem;">
          Este total (16%) es la <strong>ganancia del mes</strong>. Luego se compensa con los <strong>movimientos</strong> para obtener el saldo final.
        </div>
      </div>

      <!-- BLOQUE 2: NUEVO MOVIMIENTO + MOVIMIENTOS -->
      <div class="block">

        <h6 class="fw-bold mb-3">Nuevo movimiento manual</h6>

        <form action="cc.php?id=<?= h($driver_id) ?>&mes=<?= h($mes) ?>&anio=<?= h($anio) ?>" method="POST" enctype="multipart/form-data" class="mb-4">
          <div class="row g-2 align-items-end">
            <div class="col-6 col-md-2">
              <label class="form-label mb-1">Fecha</label>
              <input type="date" name="fecha" class="form-control form-control-sm"
                     value="<?= date('Y-m-d') ?>" required>
            </div>

            <div class="col-6 col-md-2">
              <label class="form-label mb-1">Tipo</label>
              <select name="tipo" class="form-select form-select-sm">
                <option value="adelanto">Adelanto</option>
                <option value="viaje_manual">Viaje manual</option>
                <option value="ajuste">Ajuste</option>
                <option value="reintegro">Reintegro</option>
              </select>
            </div>

            <div class="col-12 col-md-4">
              <label class="form-label mb-1">Descripción</label>
              <input type="text" name="descripcion" class="form-control form-control-sm"
                     placeholder="Ej: Adelanto en efectivo, ajuste, etc.">
            </div>

            <div class="col-6 col-md-2">
              <label class="form-label mb-1">Medio de pago</label>
              <select name="medio_pago" class="form-select form-select-sm">
                <option value="">(sin especificar)</option>
                <option value="efectivo">Efectivo</option>
                <option value="transferencia">Transferencia</option>
                <option value="cheque">Cheque</option>
                <option value="otro">Otro</option>
              </select>
            </div>

            <div class="col-3 col-md-1">
              <label class="form-label mb-1">Debe</label>
              <input
                type="text"
                name="debe"
                class="form-control form-control-sm money-input"
                value="0"
                inputmode="decimal"
                autocomplete="off"
                placeholder="0"
              >
            </div>

            <div class="col-3 col-md-1">
              <label class="form-label mb-1">Haber</label>
              <input
                type="text"
                name="haber"
                class="form-control form-control-sm money-input"
                value="0"
                inputmode="decimal"
                autocomplete="off"
                placeholder="0"
              >
            </div>

<div class="col-12 col-md-3 mt-2">
  <label class="form-label mb-1">Adjuntar archivo (opcional)</label>
  <input
    type="file"
    name="adjunto"
    class="form-control form-control-sm"
    accept="image/*,application/pdf"
  >
</div>


            <div class="col-12 col-md-2 d-grid mt-2">
              <button type="submit" class="btn btn-primary btn-sm text-white">
                <i class="bi bi-plus-lg"></i> Cargar movimiento
              </button>
            </div>
          </div>

          <div class="mt-2 text-muted" style="font-size:0.8rem;">
            Recordá: <strong>Debe</strong> = lo que se descuenta al chofer (adelantos, ajustes negativos).
            <strong>Haber</strong> = reintegros/ajustes positivos (si lo usás).
          </div>
        </form>

        <h6 class="fw-bold mb-2">Movimientos (<?= h($mes) ?>/<?= h($anio) ?>)</h6>

        <div class="table-responsive">
          <table class="table table-sm table-striped table-hover align-middle mb-0">
            <thead class="table-light">
              <tr>
                <th style="width:110px;white-space:nowrap;">Fecha</th>
                <th style="width:110px;white-space:nowrap;">Tipo</th>
                <th>Descripción</th>
                <th style="width:140px;white-space:nowrap;">Medio de pago</th>
                <th class="text-end" style="width:110px;white-space:nowrap;">Debe</th>
                <th class="text-end" style="width:110px;white-space:nowrap;">Haber</th>
                <th class="text-center" style="width:90px;white-space:nowrap;">Adjunto</th>
                <th class="text-center" style="width:90px;white-space:nowrap;">Acc.</th>
              </tr>
            </thead>
            <tbody>
            <?php if (empty($mov_mes)): ?>
              <tr>
                <td colspan="8" class="text-center text-muted">Sin movimientos cargados en este período.</td>
              </tr>
            <?php else: ?>
              <?php foreach ($mov_mes as $m): ?>
                <tr>
                  <td style="white-space:nowrap;"><?= h($m['fecha']) ?></td>
                  <td style="white-space:nowrap;"><?= h($m['tipo']) ?></td>
                  <td><div class="clamp-2"><?= h($m['descripcion']) ?></div></td>
                  <td style="white-space:nowrap;"><?= h($m['medio_pago']) ?></td>
                  <td class="text-end" style="white-space:nowrap;"><?= ((float)$m['debe'] != 0.0) ? '$ ' . number_format((float)$m['debe'], 2, ',', '.') : '-' ?></td>
                  <td class="text-end" style="white-space:nowrap;"><?= ((float)$m['haber'] != 0.0) ? '$ ' . number_format((float)$m['haber'], 2, ',', '.') : '-' ?></td>
                  <td class="text-center" style="white-space:nowrap;">
                    <?php if (!empty($m['adjunto'])):
                      $url = '../uploads/' . $m['adjunto'];
                    ?>
                      <a href="<?= h($url) ?>" target="_blank" class="btn btn-outline-secondary btn-sm" title="Ver adjunto">
                        <i class="bi bi-image"></i>
                      </a>
                    <?php else: ?>
                      <span class="text-muted">—</span>
                    <?php endif; ?>
                  </td>
                  <td class="text-center" style="white-space:nowrap;">
                    <a href="cc.php?id=<?= h($driver_id) ?>&mes=<?= h($mes) ?>&anio=<?= h($anio) ?>&delete=<?= (int)$m['id'] ?>"
                       class="btn btn-danger btn-sm text-white"
                       onclick="return confirm('¿Eliminar este movimiento?');"
                       title="Eliminar movimiento">
                      <i class="bi bi-trash"></i>
                    </a>
                  </td>
                </tr>
              <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
          </table>
        </div>

      </div><!-- /block 2 -->

    </div>
  </div>

</main>

<?php include "../assets/footer.php"; ?>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<script>
(function () {
  function formatAR(value) {
    value = (value || '').toString().replace(/[^\d,]/g, '');
    const parts = value.split(',');
    let intPart = parts[0] || '';
    let decPart = parts[1] || '';
    decPart = decPart.slice(0, 2);
    intPart = intPart.replace(/^0+(?=\d)/, '');
    intPart = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
    return decPart.length ? (intPart || '0') + ',' + decPart : (intPart || '0');
  }

  function attach(el) {
    el.value = formatAR(el.value);

    el.addEventListener('input', function () {
      const pos = el.selectionStart;
      const before = el.value;
      el.value = formatAR(el.value);
      const diff = el.value.length - before.length;
      el.setSelectionRange(Math.max(0, pos + diff), Math.max(0, pos + diff));
    });

    el.addEventListener('blur', function () {
      if (!el.value || el.value === ',') el.value = '0';
      el.value = formatAR(el.value);
    });
  }

  document.querySelectorAll('.money-input').forEach(attach);

  const form = document.querySelector('form[method="POST"]');
  if (form) {
    form.addEventListener('submit', function () {
      document.querySelectorAll('.money-input').forEach(function (el) {
        if (!el.value || el.value.trim() === '') el.value = '0';
      });
    });
  }
})();
</script>

</body>
</html>