<?php
/**
 * public/dashboard.php
 */
session_start();
require_once "../config/db.php";
require_once "../helpers/functions.php";

// Verificamos que esté logueado
check_auth();

// Obtenemos la lista de choferes para el select
$drivers = $pdo->query("SELECT id, nombre, apellido FROM drivers ORDER BY nombre")->fetchAll(PDO::FETCH_ASSOC);

// Obtenemos la lista de lugares (para Origen y Destino)
$places = $pdo->query("SELECT id, nombre FROM places ORDER BY nombre")->fetchAll(PDO::FETCH_ASSOC);

// Capturamos los filtros enviados por GET (o dejamos vacío si no se selecciona nada)
$filterDriver = $_GET['driver'] ?? '';
$filterOrigin = $_GET['origin'] ?? '';
$filterDestination = $_GET['destination'] ?? '';
$filterFechaOrigen = $_GET['fecha_origen'] ?? '';
$filterFechaDestino = $_GET['fecha_destino'] ?? '';

// Asignamos fechas por defecto si no se seleccionan
if (empty($filterFechaOrigen)) {
    $filterFechaOrigen = date('Y-m-01'); // primer día del mes actual
}
if (empty($filterFechaDestino)) {
    $filterFechaDestino = date('Y-m-t');   // día actual
}

// Construimos la cláusula WHERE para la consulta de estadísticas
$conditions = [];
$params = [];

if ($filterDriver !== '') {
    $conditions[] = "driver_id = :driver_id";
    $params[':driver_id'] = $filterDriver;
}
if ($filterOrigin !== '') {
    $conditions[] = "origin_id = :origin_id";
    $params[':origin_id'] = $filterOrigin;
}
if ($filterDestination !== '') {
    $conditions[] = "destination_id = :destination_id";
    $params[':destination_id'] = $filterDestination;
}
// Aquí usamos >= y <= para trabajar con rangos de fechas
if ($filterFechaOrigen !== '') {
    $conditions[] = "fecha_origen >= :fecha_origen";
    $params[':fecha_origen'] = $filterFechaOrigen;
}
if ($filterFechaDestino !== '') {
    $conditions[] = "fecha_destino <= :fecha_destino";
    $params[':fecha_destino'] = $filterFechaDestino;
}

$whereClause = count($conditions) > 0 ? implode(" AND ", $conditions) : "1";

// Consulta única para obtener todos los datos requeridos
// Se asume que "toneladas" existe y que "tarifa_real" es la tarifa por tonelada.
$sql_stats = "
    SELECT 
        COUNT(*) AS total_trips,
        SUM(CASE WHEN estado_viaje = 'Activo' THEN 1 ELSE 0 END) AS active_trips,
        SUM(CASE WHEN estado_viaje <> 'Activo' THEN 1 ELSE 0 END) AS non_active_trips,
        SUM(CASE WHEN estado_facturacion = 'Facturado' THEN 1 ELSE 0 END) AS billed_count,
        SUM(CASE WHEN estado_facturacion = 'No Facturado' THEN 1 ELSE 0 END) AS not_billed_count,

        -- Monto total facturado (tarifa_real * toneladas) SOLO cuando está facturado
        SUM(
          CASE WHEN estado_facturacion = 'Facturado' THEN (tarifa_real * toneladas) ELSE 0 END
        ) AS total_amount_billed,

        -- Monto total no facturado (tarifa_real * toneladas) SOLO cuando no está facturado
        SUM(
          CASE WHEN estado_facturacion = 'No Facturado' THEN (tarifa_real * toneladas) ELSE 0 END
        ) AS total_amount_not_billed
    FROM trips
    WHERE $whereClause
";
$stmt_stats = $pdo->prepare($sql_stats);
$stmt_stats->execute($params);
$stats = $stmt_stats->fetch(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="es" class="h-full bg-white">
<head>
  <meta charset="UTF-8" />
  <title>Dashboard - Gestión de Camiones</title>
  <!-- Tailwind CSS -->
  <script src="https://cdn.tailwindcss.com"></script>
  <!-- jQuery (necesario para Select2) -->
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <!-- Select2 CSS -->
  <link href="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/css/select2.min.css" rel="stylesheet" />

  <!-- Estilo personalizado para fondo blanco y tipografía -->
  <style>
    body {
      background-color: #fff;
      font-family: 'Inter', sans-serif;
    }
    /* Ajustes para que el contenedor Select2 use el 100% y coincida con Tailwind */
    .select2-container {
      width: 100% !important;
    }
    .select2-selection--single {
      height: 2.5rem !important;
      border: 1px solid #d1d5db !important;
      border-radius: 0.375rem !important;
      padding: 0.5rem 0.75rem !important;
      display: flex !important;
      align-items: center !important;
      background-color: #fff !important;
    }
    .select2-selection__rendered {
      line-height: 1.25rem !important;
    }
    .select2-selection__arrow {
      height: 2.5rem !important;
      top: 0px !important;
      right: 0.75rem !important;
      display: flex !important;
      align-items: center !important;
    }
    /* Animación fadeIn desde la izquierda */
    @keyframes fadeInLeft {
      0% {
        opacity: 0;
        transform: translateX(-20px);
      }
      100% {
        opacity: 1;
        transform: translateX(0);
      }
    }
    .fadeInLeft {
      animation: fadeInLeft 0.5s ease-out forwards;
    }
  </style>
</head>
<body class="min-h-screen flex flex-col bg-white">
  
  <!-- Barra de navegación -->
  <?php include "assets/nav.php"; ?>
  
  <main class="flex-grow pt-14">
    <div class="container mx-auto px-4 py-6">
      
      <!-- Sección Estadísticas con animación fadeInLeft -->
      <div class="bg-white rounded-lg shadow px-4 py-5 sm:px-6 fadeInLeft">
        <h1 class="text-lg leading-6 font-medium text-green-900 mb-4">Estadísticas</h1>
        
        <!-- Formulario de filtros -->
        <form method="GET" class="grid grid-cols-1 md:grid-cols-5 gap-4 mb-6 items-start">
          <!-- Chofer -->
          <div class="flex flex-col">
            <label for="driver" class="text-sm font-medium text-gray-700 mb-1">Chofer</label>
            <select name="driver" id="driver" class="block w-full border border-gray-300 rounded-md shadow-sm h-10">
              <option value="">Todos</option>
              <?php foreach ($drivers as $driver): ?>
                <option value="<?= $driver['id'] ?>" <?= ($filterDriver == $driver['id']) ? 'selected' : '' ?>>
                  <?= htmlspecialchars($driver['nombre'] . ' ' . $driver['apellido']) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          
          <!-- Origen -->
          <div class="flex flex-col">
            <label for="origin" class="text-sm font-medium text-gray-700 mb-1">Origen</label>
            <select name="origin" id="origin" class="block w-full border border-gray-300 rounded-md shadow-sm h-10">
              <option value="">Todos</option>
              <?php foreach ($places as $place): ?>
                <option value="<?= $place['id'] ?>" <?= ($filterOrigin == $place['id']) ? 'selected' : '' ?>>
                  <?= htmlspecialchars($place['nombre']) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          
          <!-- Destino -->
          <div class="flex flex-col">
            <label for="destination" class="text-sm font-medium text-gray-700 mb-1">Destino</label>
            <select name="destination" id="destination" class="block w-full border border-gray-300 rounded-md shadow-sm h-10">
              <option value="">Todos</option>
              <?php foreach ($places as $place): ?>
                <option value="<?= $place['id'] ?>" <?= ($filterDestination == $place['id']) ? 'selected' : '' ?>>
                  <?= htmlspecialchars($place['nombre']) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          
          <!-- Fecha de Origen -->
          <div class="flex flex-col">
            <label for="fecha_origen" class="text-sm font-medium text-gray-700 mb-1">Fecha de Origen</label>
            <input type="date" name="fecha_origen" id="fecha_origen" value="<?= htmlspecialchars($filterFechaOrigen) ?>" class="block w-full border border-gray-300 rounded-md shadow-sm h-10">
          </div>
          
          <!-- Fecha de Destino -->
          <div class="flex flex-col">
            <label for="fecha_destino" class="text-sm font-medium text-gray-700 mb-1">Fecha de Destino</label>
            <input type="date" name="fecha_destino" id="fecha_destino" value="<?= htmlspecialchars($filterFechaDestino) ?>" class="block w-full border border-gray-300 rounded-md shadow-sm h-10">
          </div>
          
          <!-- Botones en fila completa -->
          <div class="md:col-span-5 flex justify-end space-x-2 mt-2">
            <button type="submit" class="inline-flex items-center px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white text-sm font-medium rounded-md shadow">
              Filtrar
            </button>
            <a href="<?= $_SERVER['PHP_SELF'] ?>" class="inline-flex items-center px-4 py-2 bg-gray-600 hover:bg-gray-700 text-white text-sm font-medium rounded-md shadow">
              Limpiar Filtros
            </a>
          </div>
        </form>

        <!-- PRIMERA FILA: 5 contenedores (conteos) -->
        <div class="grid grid-cols-1 md:grid-cols-5 gap-4">
          <div class="bg-white p-4 rounded shadow">
            <p class="text-gray-600">Total de Viajes</p>
            <p class="text-2xl font-bold"><?= $stats['total_trips'] ?? 0 ?></p>
          </div>
          <div class="bg-white p-4 rounded shadow">
            <p class="text-gray-600">Viajes Activos</p>
            <p class="text-2xl font-bold"><?= $stats['active_trips'] ?? 0 ?></p>
          </div>
          <div class="bg-white p-4 rounded shadow">
            <p class="text-gray-600">Viajes No Activos</p>
            <p class="text-2xl font-bold"><?= $stats['non_active_trips'] ?? 0 ?></p>
          </div>
          <div class="bg-white p-4 rounded shadow">
            <p class="text-gray-600">Facturados</p>
            <p class="text-2xl font-bold"><?= $stats['billed_count'] ?? 0 ?></p>
          </div>
          <div class="bg-white p-4 rounded shadow">
            <p class="text-gray-600">No Facturados</p>
            <p class="text-2xl font-bold"><?= $stats['not_billed_count'] ?? 0 ?></p>
          </div>
        </div>

        <!-- SEGUNDA FILA: 2 contenedores (montos) -->
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
          <!-- Monto total facturado (tarifa_real * toneladas) -->
          <div class="bg-white p-4 rounded shadow">
            <p class="text-gray-600">Monto Total de Viajes Facturados</p>
            <p class="text-2xl font-bold">
              $ <?= $stats['total_amount_billed'] ? number_format($stats['total_amount_billed'], 2) : '0.00' ?>
            </p>
          </div>
          <!-- Monto total NO facturado (tarifa_real * toneladas) -->
          <div class="bg-white p-4 rounded shadow">
            <p class="text-gray-600">Monto Total de Viajes No Facturados</p>
            <p class="text-2xl font-bold">
              $ <?= $stats['total_amount_not_billed'] ? number_format($stats['total_amount_not_billed'], 2) : '0.00' ?>
            </p>
          </div>
        </div>

      </div>
    </div>
  </main>
  
  <?php include "assets/footer.php"; ?>

  <!-- Select2 JS -->
  <script src="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js"></script>
  <script>
    $(document).ready(function(){
      $('#driver, #origin, #destination').select2({
        placeholder: "Seleccione una opción",
        allowClear: true
      });
    });
  </script>
</body>
</html>
