<?php
session_start();
require_once "../../config/db.php";
require_once "../../helpers/functions.php";
check_auth();
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Calculadora</title>

  <!-- Bootstrap + Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css">

  <style>
    :root{
      --nav-bg:#1f2b35;
      --ok:#198754;          /* verde (ganador) */
      --warn:#f68b1e;        /* naranja */
      --bad:#dc3545;         /* rojo (perdedor/negativo) */
      --primary:#0d6efd;     /* azul (editable) */
      --ink:#111827;         /* negro */
    }
    body{background:#f6f7f8;font-family:"Segoe UI",Arial,sans-serif;font-size:14px}
    .content-wrap{margin-top:82px;padding-bottom:40px}
    .page-card{border-radius:8px;border:1px solid #e6e9ec;background:#fff}
    .calc-label{color:#111827;font-weight:600}
    .calc-input{background:#e9f2ff;border:1px solid #cfe1ff;color:#111827;text-align:right}
    .calc-input:focus{box-shadow:none;border-color:#0d6efd;background:#e3eeff}
    .t-fixed th,.t-fixed td{vertical-align:middle}
    .neg{color:var(--bad)!important}
    .pos{color:var(--ok)!important}
    .summary-good{background:#e9f7ef;border-left:4px solid var(--ok)}
    .summary-neutral{background:#f8fafc;border-left:4px solid #cbd5e1}
    .row-title{color:#111827;font-weight:700}
    .table thead th{background:#f3f4f6}
    .subtle{color:#64748b}

    /* Columnas más juntas */
    .table>:not(caption)>*>*{padding:.35rem .5rem!important}
    .table>:not(caption)>* > :nth-child(2){border-right:1px solid #e5e7eb}
    .table>:not(caption)>* > :nth-child(3){border-left:1px solid #e5e7eb}

    .table thead th:first-child{width:30%}
    .table thead th:nth-child(2),
    .table thead th:nth-child(3){width:32%}

    @media (min-width:992px){
      .table{max-width:700px;margin:0 auto}
    }

    .table td,.table th{
      font-size:13px!important;line-height:1.2;
      padding:.35rem .45rem!important;white-space:normal!important;word-wrap:break-word;
      max-height:2.4em;overflow:hidden;
    }

    /* Filas de ganancias destacadas */
    .gain-row>td{background:#e9f7ef!important;font-weight:700;border-top:2px solid #d1e7dd;border-bottom:2px solid #d1e7dd}
    .gain-row>td:first-child{color:#0f5132}
    .gain-row .pos{color:#0f5132!important}
    .gain-row .neg{color:#842029!important}

    /* Botón rojo BORRAR */
    .btn-clear{background:#dc3545;color:#fff;border:none;padding:.25rem .5rem;border-radius:.35rem;font-weight:600}
    .btn-clear:hover{background:#b02a37;color:#fff}
  </style>
</head>
<body>
<?php include "../assets/nav2.php"; ?>

<div class="content-wrap container">
  <div class="page-card card">
    <div class="card-body">
      <div class="d-flex justify-content-between align-items-center mb-3">
        <h5 class="mb-0 text-primary fw-bold">Calculadora de Conveniencia (2 viajes)</h5>
      </div>
    </div>

    <div class="table-responsive">
      <table class="table t-fixed align-middle">
        <thead>
          <tr>
            <th class="row-title">Concepto</th>
            <th class="text-center">
              Viaje 1
              <button type="button" class="btn-clear ms-2" data-clear="a">Borrar</button>
            </th>
            <th class="text-center">
              Viaje 2
              <button type="button" class="btn-clear ms-2" data-clear="b">Borrar</button>
            </th>
          </tr>
        </thead>

        <tbody id="calcBody">
          <!-- ENTRADAS -->
          <tr><td class="calc-label">Kilómetros</td>
            <td><input id="a_km" class="form-control form-control-sm calc-input" type="number" step="0.01" value="1000"></td>
            <td><input id="b_km" class="form-control form-control-sm calc-input" type="number" step="0.01" value="900"></td>
          </tr>
          <tr><td class="calc-label">Toneladas</td>
            <td><input id="a_tn" class="form-control form-control-sm calc-input" type="number" step="0.01" value="36"></td>
            <td><input id="b_tn" class="form-control form-control-sm calc-input" type="number" step="0.01" value="36"></td>
          </tr>
          <tr><td class="calc-label">Tarifa real ($/tn)</td>
            <td><input id="a_tarifa" class="form-control form-control-sm calc-input" type="number" step="0.01" value="28000"></td>
            <td><input id="b_tarifa" class="form-control form-control-sm calc-input" type="number" step="0.01" value="20000"></td>
          </tr>
          <tr><td class="calc-label">Peajes / Varios ($)</td>
            <td><input id="a_peajes" class="form-control form-control-sm calc-input" type="number" step="0.01" value="20000"></td>
            <td><input id="b_peajes" class="form-control form-control-sm calc-input" type="number" step="0.01" value="0"></td>
          </tr>
          <tr><td class="calc-label">Precio Comb ($/Lt)</td>
            <td><input id="a_pcomb" class="form-control form-control-sm calc-input" type="number" step="0.01" value="1520"></td>
            <td><input id="b_pcomb" class="form-control form-control-sm calc-input" type="number" step="0.01" value="1520"></td>
          </tr>
          <tr><td class="calc-label">Consumo (km/Lt)</td>
            <td><input id="a_kmplt" class="form-control form-control-sm calc-input" type="number" step="0.01" value="2.6"></td>
            <td><input id="b_kmplt" class="form-control form-control-sm calc-input" type="number" step="0.01" value="2.6"></td>
          </tr>
          <tr><td class="calc-label">Comisión %</td>
            <td><input id="a_com_pct" class="form-control form-control-sm calc-input" type="number" step="0.01" value="10"></td>
            <td><input id="b_com_pct" class="form-control form-control-sm calc-input" type="number" step="0.01" value="10"></td>
          </tr>
          <tr><td class="calc-label">Comisión fija ($)</td>
            <td><input id="a_com_fija" class="form-control form-control-sm calc-input" type="number" step="0.01" value="20000"></td>
            <td><input id="b_com_fija" class="form-control form-control-sm calc-input" type="number" step="0.01" value="10000"></td>
          </tr>
          <tr><td class="calc-label">Chofer %</td>
            <td><input id="a_chofer_pct" class="form-control form-control-sm calc-input" type="number" step="0.01" value="16"></td>
            <td><input id="b_chofer_pct" class="form-control form-control-sm calc-input" type="number" step="0.01" value="16"></td>
          </tr>

          <!-- RESULTADOS -->
          <tr class="table-light"><td colspan="3" class="fw-bold">Resultados</td></tr>
          <tr><td class="calc-label">Neto</td>
            <td id="a_neto" class="text-end"></td>
            <td id="b_neto" class="text-end"></td>
          </tr>
          <tr><td class="calc-label">IVA (21%)</td>
            <td id="a_iva" class="text-end subtle"></td>
            <td id="b_iva" class="text-end subtle"></td>
          </tr>
          <tr><td class="calc-label">Total con IVA</td>
            <td id="a_total_iva" class="text-end"></td>
            <td id="b_total_iva" class="text-end"></td>
          </tr>
          <tr><td class="calc-label">Peajes / Varios ($)</td>
            <td id="a_peajes_val" class="text-end" style="color:var(--warn)"></td>
            <td id="b_peajes_val" class="text-end" style="color:var(--warn)"></td>
          </tr>
          <tr><td class="calc-label">Comisión %</td>
            <td id="a_com_pct_val" class="text-end" style="color:var(--warn)"></td>
            <td id="b_com_pct_val" class="text-end" style="color:var(--warn)"></td>
          </tr>
          <tr><td class="calc-label">Comisión fija</td>
            <td id="a_com_fija_val" class="text-end" style="color:var(--warn)"></td>
            <td id="b_com_fija_val" class="text-end" style="color:var(--warn)"></td>
          </tr>
          <tr><td class="calc-label">Chofer %</td>
            <td id="a_chofer_val" class="text-end" style="color:var(--warn)"></td>
            <td id="b_chofer_val" class="text-end" style="color:var(--warn)"></td>
          </tr>
          <tr><td class="calc-label">Gasto de combustible (Lt)</td>
            <td id="a_litros" class="text-end subtle"></td>
            <td id="b_litros" class="text-end subtle"></td>
          </tr>
          <tr><td class="calc-label">Gasto en combustible ($)</td>
            <td id="a_gasto_comb" class="text-end" style="color:var(--warn)"></td>
            <td id="b_gasto_comb" class="text-end" style="color:var(--warn)"></td>
          </tr>

          <!-- GANANCIAS -->
          <tr class="table-light"><td colspan="3" class="fw-bold">Ganancias</td></tr>
          <tr class="gain-row"><td class="calc-label">Ganancia sin IVA</td>
            <td id="a_gan_sin" class="text-end fw-bold"></td>
            <td id="b_gan_sin" class="text-end fw-bold"></td>
          </tr>
          <tr class="gain-row"><td class="calc-label">Ganancia con IVA 50%</td>
            <td id="a_gan_50" class="text-end fw-bold"></td>
            <td id="b_gan_50" class="text-end fw-bold"></td>
          </tr>
          <tr class="gain-row"><td class="calc-label">Ganancia con IVA</td>
            <td id="a_gan_con" class="text-end fw-bold"></td>
            <td id="b_gan_con" class="text-end fw-bold"></td>
          </tr>

          <!-- RATIOS (sin ganancia x km) -->
          <tr class="table-light"><td colspan="3" class="fw-bold">Ratios por Km y Facturación</td></tr>
          <tr><td class="calc-label">Facturación por Km sin IVA</td>
            <td id="a_fact_sin" class="text-end subtle"></td>
            <td id="b_fact_sin" class="text-end subtle"></td>
          </tr>
          <tr><td class="calc-label">Facturación por Km con IVA</td>
            <td id="a_fact_con" class="text-end subtle"></td>
            <td id="b_fact_con" class="text-end subtle"></td>
          </tr>
        </tbody>
      </table>
    </div>

    <!-- Resumen comparativo -->
    <div id="summary" class="p-3 rounded mt-3 summary-neutral">
      <div class="fw-bold" id="summaryTitle">Comparativa</div>
      <div id="summaryBody" class="subtle"></div>
    </div>

  </div>
</div>

<script>
const $ = id => document.getElementById(id);
const money = (n) => isFinite(n) ? n.toLocaleString('es-AR',{style:'currency',currency:'ARS',maximumFractionDigits:2}) : '—';
const num   = (n) => isFinite(n) ? n.toLocaleString('es-AR',{maximumFractionDigits:2}) : '—';
const pct   = (n) => isFinite(n) ? n.toLocaleString('es-AR',{maximumFractionDigits:2}) + '%' : '—';

function read(prefix){
  return {
    km: parseFloat($(prefix+'_km').value)||0,
    tn: parseFloat($(prefix+'_tn').value)||0,
    tarifa: parseFloat($(prefix+'_tarifa').value)||0,
    peajes: parseFloat($(prefix+'_peajes').value)||0,
    pcomb: parseFloat($(prefix+'_pcomb').value)||0,
    kmplt: parseFloat($(prefix+'_kmplt').value)||0,
    com_pct: (parseFloat($(prefix+'_com_pct').value)||0)/100,
    com_fija: parseFloat($(prefix+'_com_fija').value)||0,
    chofer_pct: (parseFloat($(prefix+'_chofer_pct').value)||0)/100
  };
}

function calc(m){
  const neto = m.tn * m.tarifa;
  const iva = neto * 0.21;
  const total_iva = neto + iva;
  const com_pct_val = neto * m.com_pct;
  const com_fija_val = m.com_fija;
  const chofer_val = neto * m.chofer_pct;
  const litros = (m.kmplt>0) ? (m.km / m.kmplt) : 0;
  const gasto_comb = m.pcomb * litros;

  const gan_sin = neto - com_pct_val - com_fija_val - chofer_val - gasto_comb - m.peajes;
  const gan_50  = neto + (iva*0.5) - com_pct_val - com_fija_val - chofer_val - gasto_comb - m.peajes;
  const gan_con = total_iva - com_pct_val - com_fija_val - chofer_val - gasto_comb - m.peajes;

  const fact_sin = (m.km>0) ? (neto/m.km) : 0;
  const fact_con = (m.km>0) ? (total_iva/m.km) : 0;

  return {neto,iva,total_iva,com_pct_val,com_fija_val,chofer_val,litros,gasto_comb,gan_sin,gan_50,gan_con,fact_sin,fact_con};
}

function paint(id, value){
  const el=$(id);
  el.textContent = (id.includes('litros')) ? num(value) : money(value);
  el.classList.remove('neg','pos');
  if (id.includes('gan')) {
    if (value<0) el.classList.add('neg'); else el.classList.add('pos');
  }
}

function update(){
  const A = read('a');
  const B = read('b');
  const Ca = calc(A);
  const Cb = calc(B);

  // Viaje 1
  paint('a_neto', Ca.neto);
  paint('a_iva', Ca.iva);
  paint('a_total_iva', Ca.total_iva);
  paint('a_peajes_val', -A.peajes);
  paint('a_com_pct_val', -Ca.com_pct_val);
  paint('a_com_fija_val', -Ca.com_fija_val);
  paint('a_chofer_val', -Ca.chofer_val);
  $('a_litros').textContent = num(Ca.litros);
  paint('a_gasto_comb', -Ca.gasto_comb);
  paint('a_gan_sin', Ca.gan_sin);
  paint('a_gan_50', Ca.gan_50);
  paint('a_gan_con', Ca.gan_con);
  $('a_fact_sin').textContent = money(Ca.fact_sin);
  $('a_fact_con').textContent = money(Ca.fact_con);

  // Viaje 2
  paint('b_neto', Cb.neto);
  paint('b_iva', Cb.iva);
  paint('b_total_iva', Cb.total_iva);
  paint('b_peajes_val', -B.peajes);
  paint('b_com_pct_val', -Cb.com_pct_val);
  paint('b_com_fija_val', -Cb.com_fija_val);
  paint('b_chofer_val', -Cb.chofer_val);
  $('b_litros').textContent = num(Cb.litros);
  paint('b_gasto_comb', -Cb.gasto_comb);
  paint('b_gan_sin', Cb.gan_sin);
  paint('b_gan_50', Cb.gan_50);
  paint('b_gan_con', Cb.gan_con);
  $('b_fact_sin').textContent = money(Cb.fact_sin);
  $('b_fact_con').textContent = money(Cb.fact_con);

  // Resumen (basado en Ganancia sin IVA)
  const box = $('summary'), body= $('summaryBody');
  box.className='p-3 rounded mt-3 summary-neutral';
  const ganadorSin = (Ca.gan_sin > Cb.gan_sin) ? 'Viaje 1' : (Ca.gan_sin < Cb.gan_sin ? 'Viaje 2' : null);
  const perdedorSin = (ganadorSin === 'Viaje 1') ? 'Viaje 2' : (ganadorSin === 'Viaje 2' ? 'Viaje 1' : null);
  const diffSin = Math.abs(Ca.gan_sin - Cb.gan_sin);
  const baseSin = (ganadorSin === 'Viaje 1') ? Math.abs(Cb.gan_sin) : Math.abs(Ca.gan_sin);
  const diffSinPct = baseSin !== 0 ? (diffSin / baseSin) * 100 : 0;

  let html = '';
  if (ganadorSin === null) {
    html += `• Ganancia sin IVA: ${money(Ca.gan_sin)} en ambos viajes.`;
  } else {
    html += `• Según <strong>Ganancia sin IVA</strong>: ${ganadorSin} supera a ${perdedorSin} por <strong>${money(diffSin)}</strong> (${pct(diffSinPct)} sobre el perdedor).`;
  }
  body.innerHTML = html;
}

// --- Botones BORRAR (reset a 0 cada viaje) ---
const fields = ['km','tn','tarifa','peajes','pcomb','kmplt','com_pct','com_fija','chofer_pct'];
function clearTrip(prefix){
  fields.forEach(f=>{
    const el = document.getElementById(`${prefix}_${f}`);
    if(el) el.value = 0;
  });
  update();
}
document.addEventListener('click', (e)=>{
  const btn = e.target.closest('[data-clear]');
  if(!btn) return;
  clearTrip(btn.getAttribute('data-clear'));
});

// --- TAB/ENTER vertical: baja dentro de la misma columna y, al final, salta a la otra ---
function enableVerticalTab(){
  const allInputs = Array.from(document.querySelectorAll('.calc-input'));
  const colA = allInputs.filter(el => el.id.startsWith('a_'));
  const colB = allInputs.filter(el => el.id.startsWith('b_'));

  function goNext(current){
    const isA = current.id.startsWith('a_');
    const col = isA ? colA : colB;
    const otherCol = isA ? colB : colA;

    const idx = col.indexOf(current);
    const next = (idx < col.length - 1) ? col[idx + 1] : otherCol[0];

    if (next){
      next.focus();
      setTimeout(() => next.select?.(), 50);
    }
  }

  // Desktop → Tab o Enter
  function onKeyDown(e){
    const isTrigger = (e.key === 'Tab' && !e.shiftKey) || (e.key === 'Enter');
    if (!isTrigger) return;
    e.preventDefault();
    goNext(this);
  }

  // Mobile → algunos teclados no disparan keydown → usamos “input” + timeout
  function onMobileNext(e){
    // cuando el usuario termina de escribir y toca “siguiente” o “enter” en el teclado
    if (e.inputType === 'insertLineBreak') {
      e.preventDefault?.();
      goNext(this);
    }
  }

  [...colA, ...colB].forEach(el => {
    el.addEventListener('keydown', onKeyDown);
    el.addEventListener('beforeinput', onMobileNext); // iOS / Android
  });
}


// Listeners y render inicial
document.querySelectorAll('.calc-input').forEach(i => i.addEventListener('input', update));
enableVerticalTab();
update();
</script>

<?php include "../assets/footer.php"; ?>
</body>
</html>